function [diuCyc,diuStd]=diuCycleTimeTable(timeTable,startDate,finalDate)

% inputData can be an array containing a time series of Data with n time
% steps (rows) and m different variables (coloumns)

% startDate and final have to be in the following format:
% dd-mmm-yyyy HH:MM:SS (e.g. 04-May-1990 10:30:00)

% time units (in seconds)
m = 60;          % one minute in seconds
h = 60*m;   % one hour in seconds
d = 24*h;     % one day in seconds
y = 365*d;     % one year in seconds

inputData = timeTable.Variables;

varNames  = timeTable.Properties.VariableNames;
varLen    = length(varNames);

dtTotal  = size(inputData,1);
nw       = size(inputData,2);        % number of datasets
nm       = 12;                       % number of months per year


sNum      = datenum(startDate);
fNum      = datenum(finalDate);

dayNum    = fNum - sNum;
dt        = round(round(dayNum * d)/(dtTotal-1));
dtd       = d/dt; % number of time steps per day

% repare broken days
startNum  = floor(sNum);
finalNum  = ceil(fNum) - dt/d;

dtStart   = round((sNum-startNum)*d)/dt;
dtFinal   = round((finalNum-fNum)*d)/dt;

if dtStart > 0
    nanStart  = nan(dtStart,nw);
    dtTotal   = dtTotal + dtStart;
    inputData = [nanStart;inputData];
end
if dtFinal > 0
    nanFinal  = nan(dtFinal,nw);
    dtTotal   = dtTotal + dtFinal;
    inputData = [inputData;nanFinal];
end



timeData  = linspace(startNum,finalNum,dtTotal);


% sort data by month
monthCount = month(timeData);
monthData  = cell(nm,nw);

for iw = 1:nw
    for im = 1:nm
        monthData{im,iw} = inputData(monthCount == im,iw);
    end
end   

% number of days of the respective month
nday = cellfun( @(x) length(nanmean(reshape(x,dtd,[]))'), monthData, 'UniformOutput',false );
nday = cell2mat(nday);

% reshape into diurnal cycles
monthData = cellfun( @(x) reshape(x,dtd,[]), monthData, 'UniformOutput',false);

 
ns = 4;                     % number of seasons per year
nh = 2;                     % number of halfs per year
na = 1;                     % number of halfs per year

nmps = nm/ns;               % number of months per season
nmph = nm/nh;               % number of months per half year
nmpa = nm/na;               % number of months per half year


monSeas = [12 1:11];        % month order of seasons
monHalf = [10:12 1:9];      % month order of half years
monAll  = 1:12;           % month order of half years


seasNames = {'DJF' 'MAM' 'JJA' 'SON'};
halfNames = {'winter','summer'};
allNames  = 'all'; 

seasData = cell(ns,nw);
halfData = cell(nh,nw);
allData  = cell(na,nw);


for iw = 1:nw 
    for im = 1:nm
        halfData{ceil(im/nmph),iw} = [halfData{ceil(im/nmph),iw}, monthData{monHalf(im),iw}];
        seasData{ceil(im/nmps),iw} = [seasData{ceil(im/nmps),iw}, monthData{monSeas(im),iw}];
        allData{ceil(im/nmpa),iw}  = [allData{ceil(im/nmpa),iw},  monthData{monAll(im),iw}];
    end   
end

% average by diurnal cycles
seasCyc = cellfun( @(x) nanmean(x,2), seasData, 'UniformOutput',false );
halfCyc = cellfun( @(x) nanmean(x,2), halfData, 'UniformOutput',false );
allCyc  = cellfun( @(x) nanmean(x,2), allData,  'UniformOutput',false );
seasStd = cellfun( @(x) nanstd(x,0,2),  seasData, 'UniformOutput',false );
halfStd = cellfun( @(x) nanstd(x,0,2),  halfData, 'UniformOutput',false );
allStd  = cellfun( @(x) nanstd(x,0,2),  allData,  'UniformOutput',false );



rowTimes = datetime(datestr(linspace(round(sNum),(round(sNum)+1)-1/dtd,dtd)));

for i = 1:length(seasNames)
    diuCyc.(seasNames{i}) = array2timetable(cell2mat(seasCyc(i,:)),'RowTimes',rowTimes,'VariableNames',varNames);
    diuStd.(seasNames{i}) = array2timetable(cell2mat(seasStd(i,:)),'RowTimes',rowTimes,'VariableNames',varNames);
end

for i = 1:length(halfNames)
    diuCyc.(halfNames{i}) = array2timetable(cell2mat(halfCyc(i,:)),'RowTimes',rowTimes,'VariableNames',varNames);
    diuStd.(halfNames{i}) = array2timetable(cell2mat(halfStd(i,:)),'RowTimes',rowTimes,'VariableNames',varNames);
end

diuCyc.(allNames) = array2timetable(cell2mat(allCyc),'RowTimes',rowTimes,'VariableNames',varNames);
diuStd.(allNames) = array2timetable(cell2mat(allStd),'RowTimes',rowTimes,'VariableNames',varNames);




end