# -*- coding: utf-8 -*-
"""
Created on Tue Oct 07 16:35:42 2014

@author: s4493222

Code to read the following format NOAA asciis4493222
file name convention for areal average (aravg) time series:
ann=annual average
mon=monthly average
land_ocean=merged land-ocean surface temperature
land=land surface temperature
ocean=ocean surface temperature
latitudes=southern and northern limits of areal average
v=version number
yyyymm=date for the latest data

Annual data (aravg.ann.*) :
1st column = year
2nd column = anomaly of temperature (K)
3rd column = total error variance (K**2)
4th column = high-frequency error variance (K**2)
5th column = low-frequency error variance (K**2)
6th column = bias error variance (K**2)

Monthly data (aravg.mon.*) :
1st column = year
2nd column = month
3rd column = anomaly of temperature (K)
4th column = total error variance (K**2)
5th column = high-frequency error variance (K**2)
6th column = low-frequency error variance (K**2)
7th column = bias error variance (K**2)
8th column = diagnostic variable
9th column = diagnostic variable
10th column= diagnostic variable

NOTE: anomalies are based on the climatology from 1971 to 2000
"""
'''
Code to support SI for 
Reconciling the signal and noise of atmospheric warming on decadal timescales
Roger N Jones* and James H Ricketts

Victoria Institute of Strategic Economic Studies, Victoria University, Melbourne, Victoria 8001, Australia
Correspondence to: Roger N. Jones (roger.jones@vu.edu.au)

And related publications

This code made available for informational purposes under the Creative Commons Attribution 3.0 License (enabling electronic and paper copies);
'''

import datetime
import numpy as np
  

class NOAAascii(object):
  def __init__(self,fn):
    self.__datadic={}
    self.__header=['year', 'month', 'anomt', 'vart', 'hfvar', 'lfvar', 'biasvar', 'diag1', 'diag2', 'diag3']
    with open(fn,'r') as f:
      lines=f.readlines()
    for line in lines[:]:
      year, month, anomt, vart, hfvar, lfvar, biasvar, diag1, diag2, diag3= line.split()
      year=int(year) 
      month = int(month) 
      anomt=float(anomt)
      vart=float(vart)
      hfvar=float(hfvar)
      lfvar=float(lfvar)
      biasvar=float(biasvar)
      diag1=float(diag1)
      diag2=float(diag2)
      diag3=float(diag3)
      date=datetime.date(year,month,15)
      if not year in self.__datadic:
        self.__datadic[year] = {}
      if not month in self.__datadic[year]:
        self.__datadic[year][month]=[]
      self.__datadic[year][month].append((year, month, anomt, vart, hfvar, lfvar, biasvar, diag1, diag2, diag3))
          
  def monthly(self, variable='anomt', months=None, years=None, raw=False):
    result=[]
    count=0
    col = self.__header.index(variable)
    #colate variable by year and month
    for year in np.sort(self.__datadic.keys()):
      for month in np.sort(self.__datadic[year].keys()):
        
        count = len(self.__datadic[year][month])
        if variable in ['rain', 'evap']:
          value = np.sum([entry[col] for entry in self.__datadic[year][month]])
        else:
          value = np.mean([entry[col] for entry in self.__datadic[year][month]])
        if (years == None or year in years) and (months == None or month in months):
          result.append((year, month, value, count))
    if raw:
      return [r[2] for r in result]
    else:
      return result
    
  def annually(self, variable='anomt', years=None, raw=False):
    result=[]
    count=0
    col = self.__header.index(variable)
    #colate variable by year and month
    for year in np.sort(self.__datadic.keys()):
      value=0.0
      count=0
      for month in np.sort(self.__datadic[year].keys()):
        value += np.sum([entry[col] for entry in self.__datadic[year][month]])
        count += len(self.__datadic[year][month])
      if count > 0 and (years==None or year in years):
        if variable in ['rain', 'evap']:
          result.append((year, 0, value, count))
        else:
          result.append((year, 0, value/count, count))
    if raw:
      return [r[2] for r in result]
    else:
      return result

if __name__ == "__main__":
  fn="YOUR FILE NAME"
  a=NOAAascii(fn)
  print a.annually()
