#Troy Mastes 2013
# This script will perform the tests of the multiple regression methods
# against the simple energy balance model and 2 CMIP5 runs
setwd('C:/Climate/Masters2013_ESD_Supplement')

library(TTR)
source("MultipleRegressionHelpers.R")

########First our simple model demonstration###########
nino34<-getNino34()
nino34.ts<-window(nino34, start=c(1954,9), end=2012)
sot.ts<-getSOT()
sunspots.ts<-getSunspots()

#EBM parameters

dt<-30*24*60*60 #s/mon 30 (days/mon) *24 (hrs/day)*60 (mins/hr) *60 (s/min)
years<-57
modelLength<-12*years

#forcing
anthroForcing<-1:(years*12)/(years*12)
volcForcing<-window(sot.ts,start=1955)*-23
solarForcing<-window(sunspots.ts, start=1955, end=c(2012,12))/117/4*.7
cyclicForcing<-window(ts(cos(1:(120*12)/(60*12)*2*pi), start=1945, freq=12), start=1955)

#running one-box energy balance model
runModel<-function(incAnthro, incVolc, incSolar, incENSO, incCyclic, config=0)
{
	#underlying 0.13K/dec antrho, LF 0.11K amp, ~0.23 K/dec from 1979-1990, near standstill in last decade (2003-2012)
	if (config == 0)
	{
		Cp<-4.186*1000*1000*70 #J/K (4.186 J/g/K) * (1000 g/kg) * 1000 (kg/m^3) * 70 (m^3)
		lamda<-2.0
		ninoFactor<-10
		cyclicFactor<-30
		anthroFactor<-1.5
	}
	if (config == 1) #low sensitivity, low lag time
	{
		Cp<-4.186*1000*1000*35 #J/K (4.186 J/g/K) * (1000 g/kg) * 1000 (kg/m^3) * 35 (m^3)
		lamda<-4.0
		ninoFactor<-5
		cyclicFactor<-65
		anthroFactor<-3.0
	}
	if (config ==2) #high sensitivity, high lag time
	{
		Cp<-4.186*1000*1000*120 #J/K (4.186 J/g/K) * (1000 g/kg) * 1000 (kg/m^3) * 120 (m^3)
		lamda<-1.0
		ninoFactor<-15
		cyclicFactor<-30
		anthroFactor<-0.8
	}

	T<-numeric(modelLength)
	N<-numeric(modelLength)
	for (mon in 2:modelLength)
	{
		radForcing<-0
		if (incAnthro) radForcing<-radForcing+anthroForcing[mon]*anthroFactor
		if (incVolc) radForcing<-radForcing+volcForcing[mon] 	
		if (incSolar) radForcing<-radForcing+solarForcing[mon] 
		mlFlux<-0
		if (incENSO) mlFlux<-diff(nino34.ts)[mon]*ninoFactor
		if (incCyclic) mlFlux<-mlFlux + diff(cyclicForcing)[mon]*cyclicFactor
		N[mon]<-radForcing-lamda*T[mon-1]
		T[mon]<-(radForcing+mlFlux+Cp/dt*T[mon-1])/(Cp/dt + lamda)
	}
	T
}

#will need to run 3x with different configs to populate these lists for a figure below
solarRecon<-list() 
volcRecon<-list()

#################Run specific configurations below this line################

CUR_CONFIG<-0

#sanity check of all configurations
#ts.plot(ts(runModel(TRUE, TRUE, TRUE, TRUE, TRUE,0), freq=12, start=1955),
#		ts(runModel(TRUE, TRUE, TRUE, TRUE, TRUE,1), freq=12, start=1955),
#		ts(runModel(TRUE, TRUE, TRUE, TRUE, TRUE,2), freq=12, start=1955),
#		col=c("black", "red", "blue"), xlim=c(1979,2012))


T1.ts<-ts(runModel(TRUE, TRUE, TRUE, TRUE, FALSE,CUR_CONFIG), freq=12, start=1955) #everything except LF cycle
T2.ts<-ts(runModel(TRUE, FALSE, FALSE, FALSE, FALSE,CUR_CONFIG), freq=12, start=1955) #anthropogenic only
T3.ts<-ts(runModel(TRUE, TRUE, TRUE, TRUE, TRUE,CUR_CONFIG), freq=12, start=1955) #everything plus LF cycle
T4.ts<-ts(runModel(TRUE, FALSE, FALSE, FALSE, TRUE,CUR_CONFIG), freq=12, start=1955) #anthropogenic plus cycle only

startMonth<-1979
endMonth<-c(2012,12)

##################################
#Figure 1...Straight-line anthro, try reconstructions

par(mfrow=c(3,1))
par(mar=c(2,5,0.5,2))

#reconstruction for linear regression only
lr<-runMultipleRegressions(T1.ts, nino34.ts, sot.ts, sunspots.ts, startMonth, endMonth)
recon1.ts<-T1.ts-lag(nino34.ts, -1*lr$ensoLag)*lr$ensoCoef-lag(sot.ts,-1*lr$volcLag)*lr$volcCoef-lag(sunspots.ts,-1*lr$solarLag)*lr$solarCoef

#reconstruction using exponentialDecay
ed<-runMultipleRegressionsExponentialLagged(T1.ts, nino34.ts, volcForcing, solarForcing, startMonth, endMonth)
recon2.ts<-T1.ts-getExpLaggedForcing(volcForcing+solarForcing, ed$t0Best,1955)*ed$coefForc-lag(nino34.ts, -1*ed$ensoLag)*ed$ensoCoef

#draw top image
ts.plot(T1.ts-mean(T1.ts), xlab="Model Time", ylab=expression(Delta~T~' (K)'), xlim=c(1979,2012), lty=2, ylim=c(-0.3, 0.6))
lines(T2.ts-mean(T2.ts), col="black")
lines(recon1.ts-mean(recon1.ts), col="red")
lines(recon2.ts-mean(recon2.ts), col="blue")

#draw volcanic contribtion
#volcanic
sotInfLR.ts<-window(lag(sot.ts,-1*lr$volcLag)*lr$volcCoef,start=startMonth,end=endMonth)
sotInfED.ts<-window(getExpLaggedForcing(volcForcing, ed$t0Best,1955)*ed$coefForc, start=startMonth, end=endMonth)
sotActual.ts<-window(ts(runModel(FALSE, TRUE, FALSE, FALSE, FALSE,CUR_CONFIG), freq=12, start=1955), start=startMonth, end=endMonth)
ts.plot(sotActual.ts-mean(sotActual.ts), sotInfLR.ts-mean(sotInfLR.ts), sotInfED.ts-mean(sotInfED.ts), col=c("black", "red", "blue"), ylab=expression(Delta~T~' (K)'))

#solar
solarInfLR.ts<-window(lag(sunspots.ts,-1*lr$solarLag)*lr$solarCoef,start=startMonth,end=endMonth)
solarInfED.ts<-window(getExpLaggedForcing(solarForcing, ed$t0Best,1955)*ed$coefForc, start=startMonth, end=endMonth)
solarActual.ts<-window(ts(runModel(FALSE, FALSE, TRUE, FALSE, FALSE,CUR_CONFIG), freq=12, start=1955), start=startMonth, end=endMonth)
ts.plot(solarActual.ts-mean(solarActual.ts), solarInfLR.ts-mean(solarInfLR.ts), solarInfED.ts-mean(solarInfED.ts), col=c("black", "red", "blue"), ylab=expression(Delta~T~' (K)'))

#print out these trends
print(trend(T1.ts, start=2000, end=endMonth))
print(trend(T2.ts, start=2000, end=endMonth))
print(trend(recon1.ts, start=2000, end=endMonth))
print(trend(recon2.ts, start=2000, end=endMonth))


##################################
#Figure 2...anthro + cyclic, try reconstructions

par(mfrow=c(3,1))
par(mar=c(2,5,0.5,2))

#reconstruction for linear regression only
lr<-runMultipleRegressions(T3.ts, nino34.ts, sot.ts, sunspots.ts, startMonth, endMonth)
recon1.ts<-T3.ts-lag(nino34.ts, -1*lr$ensoLag)*lr$ensoCoef-lag(sot.ts,-1*lr$volcLag)*lr$volcCoef-lag(sunspots.ts,-1*lr$solarLag)*lr$solarCoef

#reconstruction using exponentialDecay
ed<-runMultipleRegressionsExponentialLagged(T3.ts, nino34.ts, volcForcing, solarForcing, startMonth, endMonth)
recon2.ts<-T3.ts-getExpLaggedForcing(volcForcing+solarForcing, ed$t0Best,1955)*ed$coefForc-lag(nino34.ts, -1*ed$ensoLag)*ed$ensoCoef

#draw top image
ts.plot(T3.ts-mean(T3.ts), xlab="Model Time", ylab=expression(Delta~T~' (K)'), xlim=c(1979,2012), lty=2, ylim=c(-0.3, 0.6))
lines(T4.ts-mean(T4.ts), col="black")
lines(recon1.ts-mean(recon1.ts), col="red")
lines(recon2.ts-mean(recon2.ts), col="blue")

#draw volcanic contribtion
#volcanic
sotInfLR.ts<-window(lag(sot.ts,-1*lr$volcLag)*lr$volcCoef,start=startMonth,end=endMonth)
sotInfED.ts<-window(getExpLaggedForcing(volcForcing, ed$t0Best,1955)*ed$coefForc, start=startMonth, end=endMonth)
sotActual.ts<-window(ts(runModel(FALSE, TRUE, FALSE, FALSE, FALSE,CUR_CONFIG), freq=12, start=1955), start=startMonth, end=endMonth)
ts.plot(sotActual.ts-mean(sotActual.ts), sotInfLR.ts-mean(sotInfLR.ts), sotInfED.ts-mean(sotInfED.ts), col=c("black", "red", "blue"), ylab=expression(Delta~T~' (K)'))

#store this for multi-config chart later
volcRecon[[paste('actual',CUR_CONFIG)]]<-sotActual.ts-mean(sotActual.ts)
volcRecon[[paste('LR',CUR_CONFIG)]]<-sotInfLR.ts-mean(sotInfLR.ts)


#solar
solarInfLR.ts<-window(lag(sunspots.ts,-1*lr$solarLag)*lr$solarCoef,start=startMonth,end=endMonth)
solarInfED.ts<-window(getExpLaggedForcing(solarForcing, ed$t0Best,1955)*ed$coefForc, start=startMonth, end=endMonth)
solarActual.ts<-window(ts(runModel(FALSE, FALSE, TRUE, FALSE, FALSE,CUR_CONFIG), freq=12, start=1955), start=startMonth, end=endMonth)
ts.plot(solarActual.ts-mean(solarActual.ts), solarInfLR.ts-mean(solarInfLR.ts), solarInfED.ts-mean(solarInfED.ts), col=c("black", "red", "blue"), ylab=expression(Delta~T~' (K)'))

#store this for multi-config chart later
solarRecon[[paste('actual',CUR_CONFIG)]]<-solarActual.ts-mean(solarActual.ts)
solarRecon[[paste('LR',CUR_CONFIG)]]<-solarInfLR.ts-mean(solarInfLR.ts)


#print out these trends
print(trend(T3.ts, start=2000, end=endMonth))
print(trend(T4.ts, start=2000, end=endMonth))
print(trend(recon1.ts, start=2000, end=endMonth))
print(trend(recon2.ts, start=2000, end=endMonth))

################
#Create the figure (3) of reproducing solar and volcanic influences under different configurations
#Must run script under configs #0, 1, and 2 above first!
par(mfrow=c(2,1))
par(mar=c(2,5,0.5,2))
ts.plot(volcRecon[['actual 0']],volcRecon[['actual 1']],volcRecon[['actual 2']],
		volcRecon[['LR 0']],volcRecon[['LR 1']],volcRecon[['LR 2']],
		col=c('black','red','blue','black','red','blue'), lty=c(1,1,1,3,3,3), lwd=2)
ts.plot(solarRecon[['actual 0']],solarRecon[['actual 1']],solarRecon[['actual 2']],
		solarRecon[['LR 0']],solarRecon[['LR 1']],solarRecon[['LR 2']],
		col=c('black','red','blue','black','red','blue'), lty=c(1,1,1,3,3,3), lwd=2)
			


########Now try this method on GISS-ER run
par(mfrow=c(3,1))
par(mar=c(2,5,0.5,2))

#have to use recommended forcings that went into CMIP5 runs
tsi.ts<-getTSI_CMIP5()-mean(getTSI_CMIP5())
solarForcing<-tsi.ts/4*.7
window(sot.ts, start=2005)<-0 #no volcanic activity after this point
volcForcing<-window(sot.ts,start=1955)*-23


#Figure 5
#Using T4 to introduce "underlying trend"
giss_e2_r_run.ts<-anom(window(ts(read.table("data/historicalNat_Amon_tas_GISS-E2-R_GISS-E2-R_r1i1p1_global.txt")[[1]], start=1850, freq=12), start=1979))
giss_e2_r_nino34.ts<-anom(window(ts(read.table("data/historicalNat_Amon_tas_GISS-E2-R_GISS-E2-R_r1i1p1_nino34.txt")[[1]], start=1850, freq=12), start=1977))


#reconstruction for linear regression only
lr<-runMultipleRegressions(T4.ts+giss_e2_r_run.ts, giss_e2_r_nino34.ts, sot.ts, tsi.ts, startMonth, endMonth)
recon1.ts<-T4.ts+giss_e2_r_run.ts-lag(giss_e2_r_nino34.ts, -1*lr$ensoLag)*lr$ensoCoef-lag(sot.ts,-1*lr$volcLag)*lr$volcCoef-lag(tsi.ts,-1*lr$solarLag)*lr$solarCoef

#reconstruction using exponentialDecay
ed<-runMultipleRegressionsExponentialLagged(T4.ts+giss_e2_r_run.ts, giss_e2_r_nino34.ts, volcForcing, solarForcing, startMonth, endMonth)
recon2.ts<-T4.ts+giss_e2_r_run.ts-getExpLaggedForcing(volcForcing+solarForcing, ed$t0Best,1955)*ed$coefForc-lag(giss_e2_r_nino34.ts, -1*ed$ensoLag)*ed$ensoCoef

giss_e2_r_mean.ts<-getModelAverage("GISS-E2-R", 1850)

par(mfrow=c(3,1))
par(mar=c(2,5,0.5,2))

#draw top image
ts.plot(T4.ts+giss_e2_r_run.ts-mean(T4.ts+giss_e2_r_run.ts), xlab="Model Time", ylab=expression(Delta~T~' (K)'), xlim=c(1979,2012), lty=2, ylim=c(-0.5, 0.6))
lines(T4.ts-mean(window(T4.ts, start=startMonth, end=endMonth)), col="black")
lines(recon1.ts-mean(recon1.ts), col="red")
lines(recon2.ts-mean(recon2.ts), col="blue")

#draw reproduction of natural forcing
natInfLR.ts<-window(lag(sot.ts,-1*lr$volcLag)*lr$volcCoef+lag(tsi.ts,-1*lr$solarLag)*lr$solarCoef,start=startMonth,end=endMonth)
natInfED.ts<-window(getExpLaggedForcing(volcForcing+solarForcing, ed$t0Best,1955)*ed$coefForc, start=startMonth, end=endMonth)

ts.plot(giss_e2_r_mean.ts-mean(giss_e2_r_mean.ts), natInfLR.ts-mean(natInfLR.ts), natInfED.ts-mean(natInfED.ts), col=c("black", "red", "blue"), ylab=expression(Delta~T~' (K)'))

#approximate ENSO contribution to GISS_ER
giss_enso.ts<-giss_e2_r_run.ts-giss_e2_r_mean.ts
ensoLR.ts<-lag(giss_e2_r_nino34.ts, -1*lr$ensoLag)*lr$ensoCoef
ensoED.ts<-lag(giss_e2_r_nino34.ts, -1*ed$ensoLag)*ed$ensoCoef
ts.plot(giss_enso.ts-mean(giss_enso.ts), ensoLR.ts-mean(ensoLR.ts), ensoED.ts-mean(ensoED.ts), col=c("black", "red", "blue"), ylab=expression(Delta~T~' (K)'))

#print ENSO trends
print(trend(giss_enso.ts, start=2000, end=endMonth))
print(trend(ensoLR.ts, start=2000, end=endMonth))
print(trend(ensoED.ts, start=2000, end=endMonth))

#print out these trends
print(trend(T4.ts+giss_e2_r_run.ts, start=2000, end=endMonth))
print(trend(T4.ts, start=2000, end=endMonth))
print(trend(recon1.ts, start=2000, end=endMonth))
print(trend(recon2.ts, start=2000, end=endMonth))


########Finally try this method on CNRM-CM5 run
#Figure 6
#Using T4 to introduce "underlying trend"
cnrm_run.ts<-anom(window(ts(read.table("data/historicalNat_Amon_tas_CNRM-CM5_CNRM-CM5_r1i1p1_global.txt")[[1]], start=1850, freq=12), start=1979))
cnrm_nino34.ts<-anom(window(ts(read.table("data/historicalNat_Amon_tas_CNRM-CM5_CNRM-CM5_r1i1p1_nino34.txt")[[1]], start=1850, freq=12), start=1977))


#reconstruction for linear regression only
lr<-runMultipleRegressions(T4.ts+cnrm_run.ts, cnrm_nino34.ts, sot.ts, tsi.ts, startMonth, endMonth)
recon1.ts<-T4.ts+cnrm_run.ts-lag(cnrm_nino34.ts, -1*lr$ensoLag)*lr$ensoCoef-lag(sot.ts,-1*lr$volcLag)*lr$volcCoef-lag(tsi.ts,-1*lr$solarLag)*lr$solarCoef

#reconstruction using exponentialDecay
ed<-runMultipleRegressionsExponentialLagged(T4.ts+cnrm_run.ts, cnrm_nino34.ts, volcForcing, solarForcing, startMonth, endMonth)
recon2.ts<-T4.ts+cnrm_run.ts-getExpLaggedForcing(volcForcing+solarForcing, ed$t0Best,1955)*ed$coefForc-lag(cnrm_nino34.ts, -1*ed$ensoLag)*ed$ensoCoef

cnrm_mean.ts<-getModelAverage("CNRM-CM5", 1850)

par(mfrow=c(3,1))
par(mar=c(2,5,0.5,2))

#draw top image
ts.plot(T4.ts+cnrm_run.ts-mean(T4.ts+cnrm_run.ts), xlab="Model Time", ylab=expression(Delta~T~' (K)'), xlim=c(1979,2012), lty=2, ylim=c(-0.5, 0.6))
lines(T4.ts-mean(window(T4.ts, start=startMonth, end=endMonth)), col="black")
lines(recon1.ts-mean(recon1.ts), col="red")
lines(recon2.ts-mean(recon2.ts), col="blue")

#draw reproduction of natural forcing
natInfLR.ts<-window(lag(sot.ts,-1*lr$volcLag)*lr$volcCoef+lag(tsi.ts,-1*lr$solarLag)*lr$solarCoef,start=startMonth,end=endMonth)
natInfED.ts<-window(getExpLaggedForcing(volcForcing+solarForcing, ed$t0Best,1955)*ed$coefForc, start=startMonth, end=endMonth)

ts.plot(cnrm_mean.ts-mean(cnrm_mean.ts), natInfLR.ts-mean(natInfLR.ts), natInfED.ts-mean(natInfED.ts), col=c("black", "red", "blue"), ylab=expression(Delta~T~' (K)'))

#approximate ENSO contribution to cnrm-cm5
cnrm_enso.ts<-cnrm_run.ts-cnrm_mean.ts
ensoLR.ts<-lag(cnrm_nino34.ts, -1*lr$ensoLag)*lr$ensoCoef
ensoED.ts<-lag(cnrm_nino34.ts, -1*ed$ensoLag)*ed$ensoCoef
ts.plot(cnrm_enso.ts-mean(cnrm_enso.ts), ensoLR.ts-mean(ensoLR.ts), ensoED.ts-mean(ensoED.ts), col=c("black", "red", "blue"), ylab=expression(Delta~T~' (K)'))

#print ENSO trends
print(trend(cnrm_enso.ts, start=2000, end=endMonth))
print(trend(ensoLR.ts, start=2000, end=endMonth))
print(trend(ensoED.ts, start=2000, end=endMonth))

#print out these trends
print(trend(T4.ts+cnrm_run.ts, start=2000, end=endMonth))
print(trend(T4.ts, start=2000, end=endMonth))
print(trend(recon1.ts, start=2000, end=endMonth))
print(trend(recon2.ts, start=2000, end=endMonth))