#Troy Masters 2013

#Calculate the anomaly based on the yearly frequency
anom<-function(x, freq=12)
{
	totalSum<-numeric(freq)
	numObs<-numeric(freq)
	for (mon in 1:length(x))
	{
		totalSum[mon %% freq + 1]<-totalSum[mon %% freq + 1] + x[mon]
		numObs[mon %% freq + 1]<-numObs[mon %% freq + 1] + 1
	}
	anom<-numeric(length(x))
	for (mon in 1:length(x))
	{
		anom[mon]<-x[mon]-(totalSum[mon %% freq + 1]/numObs[mon %% freq + 1])	
	}	
	anom<-ts(anom,start=tsp(x)[1],freq=freq) 
}

#Determine the coefficients and lags from the ENSO index
#using either a linear or quadratic underlying trend
findENSOEffects<-function(temp.ts, enso.ts, startMonth, endMonth, includeQuad=FALSE)
{
	bestLag<-0
	bestCoef<-0
	adj_rsquaredMax<-0
	obs.ts<-window(temp.ts, start=startMonth, end=endMonth)
	for (ensoLag in 1:12)
    {
		enso.lagged.ts<-lag(enso.ts, -1*(ensoLag -1))
		x<-time(obs.ts)
		x2<-time(obs.ts)^2
		if (includeQuad) 
		{
			obs.lm<-lm(obs.ts ~ enso.lagged.ts + x + x2, data=ts.union(obs.ts, enso.lagged.ts, x, x2))
		}
		if (!includeQuad) 
		{
			obs.lm<-lm(obs.ts ~ enso.lagged.ts + x, data=ts.union(obs.ts, enso.lagged.ts, x))
		}
		if (summary(obs.lm)$adj.r.squared > adj_rsquaredMax) 
		{
			adj_rsquaredMax<-summary(obs.lm)$adj.r.squared
			bestLag<-ensoLag-1
			bestCoef<-coef(obs.lm)[2]
		}
	}
	list('r_squared'=adj_rsquaredMax, 'bestLag'=bestLag, 'bestCoef'=bestCoef)
}

##Traditional multiple regression a la FR11 (but without Fourier series)
runMultipleRegressions<-function(temp.ts, enso.ts, volc.ts, solar.ts, startMonth, endMonth)
{
  maxLag<-10 #set the maximum monthly lag to try
  
  adj_rsquared<-numeric((maxLag+1)^3)
  trends<-numeric((maxLag+1)^3)
  intercepts<-numeric((maxLag+1)^3)
  sotLags<-numeric((maxLag+1)^3)
  ensoLags<-numeric((maxLag+1)^3)
  solarLags<-numeric((maxLag+1)^3)
  coefSot<-numeric((maxLag+1)^3)
  coefEnso<-numeric((maxLag+1)^3)
  coefSolar<-numeric((maxLag+1)^3)
  index<-1
  
  obs.ts<-window(temp.ts, start=startMonth, end=endMonth) 
  
  hiIndex<-1
  for (sotLag in 1:(maxLag+1))
  {
    for (ensoLag in 1:(maxLag+1))
    {
      for (solarLag in 1:(maxLag+1))
      {
        sot.lagged.ts<-lag(volc.ts, -1*(sotLag-1))
        enso.lagged.ts<-lag(enso.ts, -1*(ensoLag -1))
        solar.lagged.ts<-lag(solar.ts, -1*(solarLag -1))
        
        obs.lm<-lm(obs.ts ~ time(obs.ts) + sot.lagged.ts + enso.lagged.ts + solar.lagged.ts, 
                   data=ts.union(obs.ts, sot.lagged.ts, enso.lagged.ts, solar.lagged.ts))
        
        adj_rsquared[index]<-summary(obs.lm)$adj.r.squared
        if (adj_rsquared[index] > adj_rsquared[hiIndex]) 
	{
		hiIndex<-index
	}
        trends[index]<-coef(obs.lm)[2]
        intercepts[index]<-coef(obs.lm)[1]
        sotLags[index]<-sotLag-1
        ensoLags[index]<-ensoLag-1
        solarLags[index]<-solarLag-1
        coefSolar[index]<-coef(obs.lm)[5]
        coefSot[index]<-coef(obs.lm)[3]
        coefEnso[index]<-coef(obs.lm)[4]
        index<-index+1
	
      }
    }
  }
  ret<-list('trend'=trends[hiIndex]*12, 'volcLag'=sotLags[hiIndex], 'volcCoef'=coefSot[hiIndex], 'ensoLag'=ensoLags[hiIndex], 'ensoCoef'=coefEnso[hiIndex], 'solarLag'=solarLags[hiIndex], 'solarCoef'=coefSolar[hiIndex], 'intercept'=intercepts[hiIndex])
}

#Get the exponential lag of the forcing
getExpLaggedForcing<-function(forcing, t0, startYr)
{
  monthlyN <- 1:length(forcing)
  e1 <- rev(exp(-monthlyN/t0))
  e1 <- e1 / sum(e1)
  forc.lagged.ts <- ts(convolve( forcing, e1, type='o' ), start=startYr,freq=12)
}

##Newer method with combined solar+volcanic and exponential term
runMultipleRegressionsExponentialLagged<-function(temp.ts, enso.ts, volcForcing, solarForcing, startMonth, endMonth)
{  
  maxEnsoLag<-10
  maxT0<-100
  adj_rsquaredMax<-0
  
  index<-1
  
  obs.ts<-window(temp.ts, start=startMonth, end=endMonth) 
  
  for (ensoLag in 1:(maxEnsoLag+1))
  {
	enso.lagged.ts<-lag(enso.ts, -1*(ensoLag -1))
	for (t0 in 1:100)
    {
		forc.lagged.ts<-getExpLaggedForcing(volcForcing+solarForcing, t0, 1955)
		        
        obs.lm<-lm(obs.ts ~ time(obs.ts) + forc.lagged.ts + enso.lagged.ts, 
                   data=ts.union(obs.ts, forc.lagged.ts, enso.lagged.ts))
        
        adj_rsquared<-summary(obs.lm)$adj.r.squared
        if (adj_rsquared > adj_rsquaredMax) 
		{
			adj_rsquaredMax<-adj_rsquared
			trend<-coef(obs.lm)[2]
			intercept<-coef(obs.lm)[1]
			ensoLagBest<-ensoLag-1
			coefEnso<-coef(obs.lm)[4]
			coefForc<-coef(obs.lm)[3]
			t0Best<-t0
		}
    }
  }
  ret<-list('trend'=trend, 'coefForc'=coefForc, 't0Best'=t0Best, 'ensoLag'=ensoLagBest, 'ensoCoef'=coefEnso)
}
#get full list of files
fileList<-list.files(path="data", full.names=TRUE)

#just get trend coefficient (in K/decade) from time series and start
trend<-function(x, start, end=2015)
{
	test.ts<-window(x,start=start, end=end)
	test.lm<-lm(test.ts ~ time(test.ts))
	c(coef(test.lm)[2]*100, confint(test.lm)[4]*100-coef(test.lm)[2]*100)  #convert to K/century
}

#graph and calculate the ensemble mean for that model
getModelAverage<-function(model, startDate)
{
	needNew<-TRUE
	color="yellow"
	mean.ts<-0
	count<-0
	for (i in 1:length(fileList))
	{	#find this specific model global tas files
		if (length(grep(model, fileList[i])) > 0 && length(grep("global", fileList[i])) > 0)
		{
			print(fileList[i])
			tas<-ts(read.table(fileList[i])[[1]], start=startDate, freq=12)
			tas.anom<-anom(window(tas, start=1979))
			if (needNew) {
				ts.plot(tas.anom, col=color)
				needNew<-FALSE
			} else {
				lines(tas.anom, col=color)
			}
			count<-count+1
			mean.ts<-mean.ts+tas.anom
		}
	}	
	mean.ts<-window(mean.ts/count, end=c(2012,12))
	lines(mean.ts, col="black", lwd=2)
	print(paste(count, trend(mean.ts, 1979), trend(mean.ts, 1996),	trend(mean.ts, 2000)))
	mean.ts
}

##################################
# Helper function to convert from table 
# received from Climate Explorer to time
# series
##################################
getClimateExplorerTs<-function(url, startMonth)
{
rawFormat<-read.table(url)
numVector<-numeric(12*nrow(rawFormat))
for (line in 1:nrow(rawFormat))
{
	for (col in 2:13)
	{
		numVector[(line-1)*12+col-1]<-rawFormat[[col]][line]
	}
}
climateExplorer.ts<-ts(numVector, start=startMonth, freq=12)
}

##################################
#function to grab the Nino3.4 data
##################################
getNino34<-function(){nino34.ts<-getClimateExplorerTs("data/inino5.dat.txt", c(1856,1))}

##################################
#Get stratospheric optical thickness
##################################
getSOT<-function()
{
rawFormat<-read.table("data/tau.line_2012.12.txt", skip=4)
#append 5 more years of blank 0s
rawVector<-c(rawFormat[[2]], numeric(60))
sot.ts<-ts(rawVector, start=c(1850,1), freq=12)
}

##################################
#Get sunspot observations
##################################
getSunspots<-function()
{
sunspots.ts<-getClimateExplorerTs("data/isunspots.dat.txt", c(1749,1))
}

##################################
#CMIP5-specific TSI
##################################
getTSI_CMIP5<-function()
{
rawFormat<-read.table("data/TSI_WLS_ann_1610_2008.txt", skip=3)
tsi<-ts(rawFormat[[2]], start=1610, freq=1)
#push forward to 2013 using 1996-2008 cycle per requirements
#1996-2008 maps to #2009-2021
tsi.annual<-ts(c(tsi, window(tsi,start=1996,end=2000)),start=1610)
#convert to monthly (for compatibility with other series), only get from 1955
tsi.monthly.ts<-ts(spline(window(tsi.annual, start=1954, end=2013), n=(2013-1954)*12+1)$y, start=c(1954,6), freq=12)
}


#Read giss temperatures from file
readGISS<-function()
{
	lineVector<-character(0)
	lines<-readLines("data/GISS_T.txt")
	#only get relevant rows and cols
	for (line in 1:length(lines))
	{
		if (substr(lines[line], 1, 2) == "18" | substr(lines[line], 1, 2) == "19" | substr(lines[line], 1, 2) == "20")
		{
			lines[line]<-gsub("*****", " -999", lines[line], fixed=TRUE)
			lineVector<-append(lineVector,substr(lines[line], 1, 66))
		}
	}
	rawFormat<-read.table(textConnection(lineVector))
	#convert data frame into numeric vector
	numVector<-numeric(12*nrow(rawFormat))
	for (line in 1:nrow(rawFormat))
	{
		for (col in 2:13)
		{
			numVector[(line-1)*12+col-1]<-rawFormat[[col]][line]/100
		}
	}
	closeAllConnections()
	#convert to time series
	GISS.ts<-ts(numVector,start=c(1880,1), freq=12)
}

#Read hadCRUTv4 values from file
getHadCRUT4<-function()
{
	rawFormat<-read.table("data/HadCRUT.4.2.0.0.monthly_ns_avg.txt")
	hadCRUT.ts<-ts(rawFormat[[2]],start=c(1850,1), freq=12)
}
