#Troy Masters 2012

#set working directory here
setwd("C:/Climate/Masters_2012")

library(ncdf) #for reading netCDF files

#end of period (do full length of CERES available obs)
end<-c(2011,6)

#########################
#HELPER FUNCTIONS
############################
#Calculate the anomaly based on the yearly frequency
anom<-function(x, freq=12)
{
	totalSum<-numeric(freq)
	numObs<-numeric(freq)
	for (mon in 1:length(x))
	{
		totalSum[mon %% freq + 1]<-totalSum[mon %% freq + 1] + x[mon]
		numObs[mon %% freq + 1]<-numObs[mon %% freq + 1] + 1
	}
	anom<-numeric(length(x))
	for (mon in 1:length(x))
	{
		anom[mon]<-x[mon]-(totalSum[mon %% freq + 1]/numObs[mon %% freq + 1])	
	}	
	anom<-ts(anom,start=tsp(x)[1],freq=freq) 
}

#get nicely formatted row of data based on regression of CRF against temperatures
getRegressionRow<-function(fluxAll, fluxSw, fluxLw, temps, monthsAgg)
{
        fluxAll.agg<-anom(aggregate(fluxAll, nfrequency=12/monthsAgg)/monthsAgg, 12/monthsAgg)
	fluxSw.agg<-anom(aggregate(fluxSw, nfrequency=12/monthsAgg)/monthsAgg, 12/monthsAgg)
	fluxLw.agg<-anom(aggregate(fluxLw, nfrequency=12/monthsAgg)/monthsAgg, 12/monthsAgg)
	temps.agg<-anom(aggregate(temps, nfrequency=12/monthsAgg)/monthsAgg, 12/monthsAgg)	

	regNet.lm<-lm(fluxAll.agg ~ temps.agg, data=ts.union(fluxAll.agg, temps.agg))
	regSw.lm<-lm(fluxSw.agg ~ temps.agg, data=ts.union(fluxSw.agg, temps.agg))
	regLw.lm<-lm(fluxLw.agg ~ temps.agg, data=ts.union(fluxLw.agg, temps.agg))

	#output our R^2 values to console
	print(summary(regNet.lm)$r.squared)
	print(summary(regSw.lm)$r.squared)
	print(summary(regLw.lm)$r.squared)

	y1<-paste(round(coef(regNet.lm)[2],2), "", round(coef(regNet.lm)[2]-confint(regNet.lm)[2,1], 2)) 
	y2<-paste(round(coef(regSw.lm)[2],2), "", round(coef(regSw.lm)[2]-confint(regSw.lm)[2,1], 2))
	y3<-paste(round(coef(regLw.lm)[2],2), "", round(coef(regLw.lm)[2]-confint(regLw.lm)[2,1], 2))
	regRow<-c(y1, y2, y3)
}

#perform only for the LW component
getRegressionRowLw<-function(fluxLw, temps, monthsAgg)
{
        fluxLw.agg<-anom(aggregate(fluxLw, nfrequency=12/monthsAgg)/monthsAgg, 12/monthsAgg)
	temps.agg<-anom(aggregate(temps, nfrequency=12/monthsAgg)/monthsAgg, 12/monthsAgg)	
	regLw.lm<-lm(fluxLw.agg ~ temps.agg, data=ts.union(fluxLw.agg, temps.agg))

	#output our R^2 values to console
	print(summary(regLw.lm)$r.squared)

	y3<-paste(round(coef(regLw.lm)[2],2), "", round(coef(regLw.lm)[2]-confint(regLw.lm)[2,1], 2))
	regRow<-c(NA, NA, y3)
}


###########################################
#1. PERFORM ANALYSIS OVER TERRA PERIOD
###########################################
start<-c(2000,3)

terraStart<-c(2000,3)

terra.nc<-open.ncdf("CERES_SSF1deg-Month-lite_Terra_Ed2.6_Subset_200003-201106.nc")
terra_net_all<-ts(get.var.ncdf(terra.nc, "gtoa_net_all_mon"), start=terraStart, freq=12)
terra_net_clr<-ts(get.var.ncdf(terra.nc, "gtoa_net_clr_mon"), start=terraStart, freq=12)
terra_sw_all<-ts(get.var.ncdf(terra.nc, "gtoa_sw_all_mon"), start=terraStart, freq=12)
terra_sw_clr<-ts(get.var.ncdf(terra.nc, "gtoa_sw_clr_mon"), start=terraStart, freq=12)
terra_lw_all<-ts(get.var.ncdf(terra.nc, "gtoa_lw_all_mon"), start=terraStart, freq=12)
terra_lw_clr<-ts(get.var.ncdf(terra.nc, "gtoa_lw_clr_mon"), start=terraStart, freq=12)
terra_sw_inc<-ts(get.var.ncdf(terra.nc, "gtoa_solar_all_mon"), start=terraStart, freq=12)
close.ncdf(terra.nc)

#get ERA clear-sky data
eraData<-read.table("era_flux_data.txt", header=TRUE)
era_sw_inc<-ts(eraData["sw_inc"], freq=12, start=2000)
era_sw_clr<-ts(eraData["sw_out_clr"], freq=12, start=2000)
era_lw_clr<-ts(eraData["lw_out_clr"], freq=12, start=2000)

####Calculate CRF values
terra_crf_net<-window(terra_net_all-terra_net_clr, start=start, end=end)
terra_crf_net.anom<-anom(terra_crf_net)
terra_crf_sw.anom<-anom(window(terra_sw_clr-terra_sw_all, start=start, end=end),12)
terra_crf_lw.anom<-anom(window(terra_lw_clr-terra_lw_all, start=start, end=end),12)

#combine just sw and lw to avoid incident solar inconsistency
era_crf_net<-window((era_sw_clr+era_lw_clr)-(terra_sw_all+terra_lw_all), start=start, end=end)
era_crf_net.anom<-anom(era_crf_net, 12)
era_crf_unadj<-era_crf_net.anom #store for later
era_crf_sw.anom<-anom(window(era_sw_clr-terra_sw_all, start=start, end=end),12)
era_crf_lw.anom<-anom(window(era_lw_clr-terra_lw_all, start=start, end=end),12)

##Get our temperature records
GISS.ts<-ts(read.table("GISTemp.txt")[[2]], start=1880, freq=12)
GISS.ts<-anom(window(GISS.ts, start=start, end=end))

NCDC.ts<-ts(read.table("NCDC_Temps.txt")[[2]], start=1880, freq=12)
NCDC.ts<-anom(window(NCDC.ts, start=start, end=end))


################################
# FIGURE 1: Terra all-sky, then CERES & ERA clear-sky, then CERES & ERA CRF
#all-sky
ts.plot(anom(window(terra_net_all,start=start,end=end)), ylab=expression(Delta~R[all]~'('~Wm^{-2}~')'))
#clear-skies
ts.plot(anom(window(-1*(era_sw_clr+era_lw_clr),start=start,end=end)), anom(window(terra_net_clr,start=start,end=end)), 
	col=c("red", "black"), ylab=expression(Delta~R[clr]~'('~Wm^{-2}~')'))
#CRF
ts.plot(era_crf_net.anom, terra_crf_net.anom, col=c("red", "black"),
	ylab=expression(~Delta~CRF~' ('~Wm^{-2}~')'), lwd=1)

################################
# TABLE 1 Estimates
GISS_CERES.monthly<-getRegressionRow(terra_crf_net.anom, terra_crf_sw.anom, terra_crf_lw.anom, GISS.ts, 1)
GISS_CERES.seasonal<-getRegressionRow(terra_crf_net.anom, terra_crf_sw.anom, terra_crf_lw.anom, GISS.ts, 3)
GISS_ERA.monthly<-getRegressionRow(era_crf_net.anom, era_crf_sw.anom, era_crf_lw.anom, GISS.ts, 1)
GISS_ERA.seasonal<-getRegressionRow(era_crf_net.anom, era_crf_sw.anom, era_crf_lw.anom, GISS.ts, 3)

NCDC_CERES.monthly<-getRegressionRow(terra_crf_net.anom, terra_crf_sw.anom, terra_crf_lw.anom, NCDC.ts, 1)
NCDC_CERES.seasonal<-getRegressionRow(terra_crf_net.anom, terra_crf_sw.anom, terra_crf_lw.anom, NCDC.ts, 3)
NCDC_ERA.monthly<-getRegressionRow(era_crf_net.anom, era_crf_sw.anom, era_crf_lw.anom, NCDC.ts, 1)
NCDC_ERA.seasonal<-getRegressionRow(era_crf_net.anom, era_crf_sw.anom, era_crf_lw.anom, NCDC.ts, 3)

##output results
tblMatrix<-rbind(c(GISS_CERES.monthly, GISS_CERES.seasonal), c(GISS_ERA.monthly, GISS_ERA.seasonal),
		c(NCDC_CERES.monthly,NCDC_CERES.seasonal), c(NCDC_ERA.monthly, NCDC_ERA.seasonal))
df<-data.frame(tblMatrix)
rownames(df)[1]<-"GISTemp and CERES_CRF"
rownames(df)[2]<-"GISTemp and ERA_CRF"
rownames(df)[3]<-"NCDC and CERES_CRF"
rownames(df)[4]<-"NCDC and ERA_CRF"
write.table(df, file="CloudFeedback_TERRA.txt",sep="\t")


###########################################
#2. PERFORM ANALYSIS OVER AQUA PERIOD
###########################################
start<-c(2002,9)
aquaStart<-c(2002,7)
aqua.nc<-open.ncdf("CERES_SSF1deg-Month-lite_Aqua_Ed2.6_Subset_200207-201106.nc") #open our nc file
aqua_net_all<-ts(get.var.ncdf(aqua.nc, "gtoa_net_all_mon"), start=aquaStart, freq=12)
aqua_net_clr<-ts(get.var.ncdf(aqua.nc, "gtoa_net_clr_mon"), start=aquaStart, freq=12)
aqua_sw_all<-ts(get.var.ncdf(aqua.nc, "gtoa_sw_all_mon"), start=aquaStart, freq=12)
aqua_sw_clr<-ts(get.var.ncdf(aqua.nc, "gtoa_sw_clr_mon"), start=aquaStart, freq=12)
aqua_lw_all<-ts(get.var.ncdf(aqua.nc, "gtoa_lw_all_mon"), start=aquaStart, freq=12)
aqua_lw_clr<-ts(get.var.ncdf(aqua.nc, "gtoa_lw_clr_mon"), start=aquaStart, freq=12)
aqua_sw_inc<-ts(get.var.ncdf(aqua.nc, "gtoa_solar_all_mon"), start=aquaStart, freq=12)
close.ncdf(aqua.nc)

####Calculate CRF values
aqua_crf_net<-window(aqua_net_all-aqua_net_clr, start=start, end=end)
aqua_crf_net.anom<-anom(aqua_crf_net)
aqua_crf_sw.anom<-anom(window(aqua_sw_clr-aqua_sw_all, start=start, end=end),12)
aqua_crf_lw.anom<-anom(window(aqua_lw_clr-aqua_lw_all, start=start, end=end),12)

##re-baseline other anomalies
terra_crf_net.anom<-anom(window(terra_crf_net.anom, start=start, end=end))
terra_crf_sw.anom<-anom(window(terra_crf_sw.anom, start=start, end=end))
terra_crf_lw.anom<-anom(window(terra_crf_lw.anom, start=start, end=end))

era_crf_net.anom<-anom(window(era_crf_net.anom, start=start, end=end))
era_crf_sw.anom<-anom(window(era_crf_sw.anom, start=start, end=end))
era_crf_lw.anom<-anom(window(era_crf_lw.anom, start=start, end=end))

GISS.ts<-anom(window(GISS.ts, start=start, end=end))
NCDC.ts<-anom(window(NCDC.ts, start=start, end=end))

#average both CERES satellites
ceres_crf_net.anom<-(aqua_crf_net.anom+terra_crf_net.anom)/2
ceres_crf_sw.anom<-(aqua_crf_sw.anom+terra_crf_sw.anom)/2
ceres_crf_lw.anom<-(aqua_crf_lw.anom+terra_crf_lw.anom)/2
ceres_lw_all<-(aqua_lw_all+terra_lw_all)/2

#Determine long-wave clear-sky from AIRS_CAM
fluxes<-read.table("AIRS_GFDL_GlobalFluxContributions_2002_9-2011_10.txt")
clear_sky.airs<-ts(fluxes[[5]]+fluxes[[7]]+fluxes[[6]]-(1:110)/110*.25, start=c(2002,9), freq=12)
ceres_airs_lw.anom<-anom(anom(window(clear_sky.airs, end=end))-anom(ceres_lw_all))

################################
# FIGURE 2: CERES & ERA clear-sky
#short-wave
ts.plot(anom(window(-1*(era_sw_clr),start=start,end=end)), anom(window(-1*(terra_sw_clr+aqua_sw_clr)/2,start=start,end=end)), 
	col=c("red", "black"), ylab=expression(Delta~R[SW~','~CLR]~'('~Wm^{-2}~')'))
#long-wave
ts.plot(anom(window(-1*(era_lw_clr),start=start,end=end)), anom(window(-1*(terra_lw_clr+aqua_lw_clr)/2,start=start,end=end)), 
	anom(window(-1*clear_sky.airs, start=start,end=end)), col=c("red", "black", "blue"), ylab=expression(Delta~R[LW~','~CLR]~'('~Wm^{-2}~')'))


################################
# TABLE 2 Estimates
GISS_CERES.monthly<-getRegressionRow(ceres_crf_net.anom, ceres_crf_sw.anom, ceres_crf_lw.anom, GISS.ts, 1)
GISS_CERES.seasonal<-getRegressionRow(ceres_crf_net.anom, ceres_crf_sw.anom, ceres_crf_lw.anom, GISS.ts, 3)
GISS_ERA.monthly<-getRegressionRow(era_crf_net.anom, era_crf_sw.anom, era_crf_lw.anom, GISS.ts, 1)
GISS_ERA.seasonal<-getRegressionRow(era_crf_net.anom, era_crf_sw.anom, era_crf_lw.anom, GISS.ts, 3)

NCDC_CERES.monthly<-getRegressionRow(ceres_crf_net.anom, ceres_crf_sw.anom, ceres_crf_lw.anom, NCDC.ts, 1)
NCDC_CERES.seasonal<-getRegressionRow(ceres_crf_net.anom, ceres_crf_sw.anom, ceres_crf_lw.anom, NCDC.ts, 3)
NCDC_ERA.monthly<-getRegressionRow(era_crf_net.anom, era_crf_sw.anom, era_crf_lw.anom, NCDC.ts, 1)
NCDC_ERA.seasonal<-getRegressionRow(era_crf_net.anom, era_crf_sw.anom, era_crf_lw.anom, NCDC.ts, 3)


GISS_AIRS.monthly<-getRegressionRowLw(ceres_airs_lw.anom, GISS.ts, 1)
GISS_AIRS.seasonal<-getRegressionRowLw(ceres_airs_lw.anom, GISS.ts, 3)
NCDC_AIRS.monthly<-getRegressionRowLw(ceres_airs_lw.anom, NCDC.ts, 1)
NCDC_AIRS.seasonal<-getRegressionRowLw(ceres_airs_lw.anom, NCDC.ts, 3)


tblMatrix<-rbind(c(GISS_CERES.monthly, GISS_CERES.seasonal), c(GISS_ERA.monthly, GISS_ERA.seasonal),
		c(GISS_AIRS.monthly, GISS_AIRS.seasonal),
		c(NCDC_CERES.monthly,NCDC_CERES.seasonal), c(NCDC_ERA.monthly, NCDC_ERA.seasonal),
		c(NCDC_AIRS.monthly, NCDC_AIRS.seasonal))
df<-data.frame(tblMatrix)
rownames(df)[1]<-"GISTemp and CERES_CRF"
rownames(df)[2]<-"GISTemp and ERA_CRF"
rownames(df)[3]<-"GISTemp and AIRS_CRF"
rownames(df)[4]<-"NCDC and CERES_CRF"
rownames(df)[5]<-"NCDC and ERA_CRF"
rownames(df)[6]<-"NCDC and AIRS_CRF"

write.table(df, file="CloudFeedback_AQUA.txt",sep="\t")

###########################################
#3. PERFORM ADJUSTMENTS OVER AQUA PERIOD
###########################################
fluxes<-read.table("AIRS_GFDL_GlobalFluxContributions_2002_9-2011_10.txt")
surfaceTempFluxAdj.airs<-ts(fluxes[[1]]-fluxes[[5]], start=c(2002,9), freq=12)
atmoTempFluxAdj.airs<-ts(fluxes[[2]]-fluxes[[6]], start=c(2002,9), freq=12)
lwWvFluxAdj.airs<-ts(fluxes[[3]]-fluxes[[7]], start=c(2002,9), freq=12)
swWvFluxAdj.airs<-ts(fluxes[[8]]-fluxes[[4]], start=c(2002,9), freq=12)

#include CO2 adjust - 0.21 Wm^2 times 0.16 for difference
CO2Adjust<--1*ts((1:106)/106*0.21 * 0.16, start=start, freq=12)

ceres_rcloud_net.anom<-anom(ceres_crf_net.anom + surfaceTempFluxAdj.airs 
	+  atmoTempFluxAdj.airs + CO2Adjust + + lwWvFluxAdj.airs + swWvFluxAdj.airs)

r_cloud.lm<-lm(ceres_rcloud_net.anom ~ GISS.ts)

################################
# FIGURE 3: CERES CRF and R_cloud
ts.plot(ceres_crf_net.anom, ceres_rcloud_net.anom, col=c("black", "red"), 
	ylab=expression(Delta~R[cloud]~','~Delta~CRF~' ('~Wm^{-2}~')'))

