//Troy Masters 2012
//KernelAdjustments2: Convert the AIRS downloaded individual HDF files
//into a single NC file with the GFDL resolution (for use to combine with GFDL kernels)

import java.io.*;
import java.util.*;


import ucar.ma2.Array;
import ucar.ma2.DataType;
import ucar.nc2.*;
import ucar.ma2.*;


public class KernelAdjustments2 {

	static int NUM_MONTHS = 110;
	static int NUM_LON = 144;//144;
	static int NUM_LAT = 90;//90;		
	static String DIR = "C:/Climate/AIRS";
		
	private static float[][] DoBoxAverageRegrid(float[][] oldGrid, int oldLonCount, int oldLatCount, int newLonCount, int newLatCount)
	{
		float[][] newGrid = new float[newLonCount][newLatCount];
		float newGridSizeX = (float) 360.0 / (float) newLonCount;
		float newGridSizeY = (float) 180.0 / (float) newLatCount;
		float oldGridSizeX = (float) 360.0 / (float) oldLonCount;
		float oldGridSizeY = (float) 180.0 / (float) oldLatCount;
		
		for (int x = 0; x < newLonCount; x++)
		{
			for (int y=0; y < newLatCount; y++)
			{
				float left=x*newGridSizeX, right = (x+1)*newGridSizeX;
				float top=90-y*newGridSizeY, bottom = 90-(y+1)*newGridSizeY;
				
				float totalCellWeight = 0;
				float totalCellIntegral = 0;
				
				for (int oldGridX = (int) Math.floor(left / oldGridSizeX); oldGridX < Math.floor(right/ oldGridSizeX); oldGridX++)
				{
					for (int oldGridY = (int) Math.floor(y*newGridSizeY / oldGridSizeY); oldGridY < Math.floor((y+1)*newGridSizeY / oldGridSizeY); oldGridY++)
					{
						float oldLeft=oldGridX*oldGridSizeX, oldRight = (oldGridX+1)*oldGridSizeX;
						float oldTop=90-y*newGridSizeY, oldBottom = 90-(oldGridY+1)*oldGridSizeY;
						
						double yWeight, xWeight;			
						xWeight = Math.min(oldRight, right) - Math.max(oldLeft, left);
						yWeight = Math.sin(Math.min(oldTop, top)*Math.PI/180.0) - Math.sin(Math.min(oldBottom, bottom)*Math.PI/180.0);
						
						if (oldGrid[oldGridX][oldGridY] > -998) //Not a missing value
						{
							totalCellIntegral += oldGrid[oldGridX][oldGridY] * xWeight * yWeight;
							totalCellWeight += xWeight * yWeight;
						}
					}
				}
				
				if (totalCellWeight > 0)  //Have non-blank values for this
				{
					newGrid[x][y] = totalCellIntegral / totalCellWeight;
				}
				else
				{
					if (x > 0)
						newGrid[x][y] = newGrid[x-1][y];
					else if (y > 0)
						newGrid[x][y] = newGrid[x][y-1];
				}
				
				
			}
		}
		return newGrid;
	}
	
	private static float[][] ReadArray2D(NetcdfFile file, String varName)
	{
		float[][]regrid = new float[NUM_LON][NUM_LAT];
		try
		{
			Variable v = file.findVariable(varName);
			Array array = v.read(new int[] {0,0}, new int[] {180, 360});
			Index index = array.getIndex();
			
			float[][] data = new float[360][180];
			for (int i=0; i < 180; i++)
			{
				for (int j=0; j < 360; j++)
				{
					
					data[j][i] = array.getFloat(index.set(i, j));
				}
			}
			regrid = DoBoxAverageRegrid(data, 360, 180, NUM_LON, NUM_LAT);
		}
		catch (Exception e)
		{
			System.err.print("Unable to read 2D array: " + e.toString());
		}
		return regrid;
	}
	
	private static float[][][] ReadArray3D(NetcdfFile file, String varName)
	{
		float[][][] regrid = new float[NUM_LON][NUM_LAT][12];
		try
		{
			Variable v = file.findVariable(varName);
			
			Array array = v.read(new int[] {0,0,0}, new int[] {12, 180,360});
			Index index = array.getIndex();
			
			for (int layer = 0; layer < 12; layer++)
			{
				float[][] data = new float[360][180];
				for (int i=0; i < 180; i++)
				{
					for (int j=0; j < 360; j++)
					{
						
						data[j][i] = array.getFloat(index.set(layer, i, j));
					}
				}
				data = DoBoxAverageRegrid(data, 360, 180, NUM_LON, NUM_LAT);
				CopyTo(regrid, data, layer);
			}
		}
		catch (Exception e)
		{
			System.err.print("Unable to read 3D array: " + e.toString());
		}
		return regrid;
	}
	
	private static void WriteArray2D(NetcdfFileWriteable ncWrite, String varName, float[][] data, int month)
	{
		try
		{
		  ArrayFloat arr = new ArrayFloat.D3(1, NUM_LAT, NUM_LON);
		  Index ima = arr.getIndex();
		  for (int i=0; i<NUM_LAT; i++) {
		    for (int j=0; j<NUM_LON; j++) {
		      arr.setFloat(ima.set(0, i, j), data[j][i]);
		     }
		   }
		  
		  int[] origin = new int[3];
		  origin[0] = month;
		  
		  ncWrite.write(varName, origin, arr);
		}
		catch (Exception e)
		{
			System.err.print("Unable to write 2D array: " + e.toString());
		}
	}
	
	private static void WriteArray3D(NetcdfFileWriteable ncWrite, String varName, float[][][] data, int month)
	{
		try
		{
		  ArrayFloat arr = new ArrayFloat.D4(1, 12, NUM_LAT, NUM_LON);
		  Index ima = arr.getIndex();

			  for (int pl=0; pl < 12; pl++) //pl = pressure level
			  {
				  for (int i=0; i<NUM_LAT; i++) {
				    for (int j=0; j<NUM_LON; j++) {
				      arr.setFloat(ima.set(0,pl, i, j), data[j][i][pl]);
				     }
				   }
			  }
		
		  int[] origin = new int[4];
		  origin[0] = month;
		  
		  ncWrite.write(varName, origin, arr);
		}
		catch (Exception e)
		{
			System.err.print("Unable to write 3D array: " + e.toString());
		}
	}
	
	/*private static void CopyTo(float[][][][] dest, float[][][] src, int mon)
	{
		for (int layer=0; layer < 12; layer++)
		{
			for (int i=0; i < 128; i++)
				for (int j=0; j < 64; j++)
					dest[i][j][layer][mon] = src[i][j][layer];
		}
	}*/
	
	private static void CopyTo(float[][][] dest, float[][] src, int mon)
	{
		for (int i=0; i < NUM_LON; i++)
			for (int j=0; j < NUM_LAT; j++)
				dest[i][j][mon] = src[i][j];
	}
	
	private static String getFile(int year, int month)
	{
		String foundFile = null;
		File dir = new File("C:/climate/airs");
		String files[] = dir.list();
		for (int i=0; i < files.length; i++)
		{
			String monthStr = (new Integer(month)).toString();
			if (monthStr.length() == 1)
				monthStr = "0" + monthStr;
			String totalString = (new Integer(year)).toString() + "." + monthStr;
			
			if (files[i].endsWith(".hdf") && files[i].contains(totalString))
			{
				return "C:/climate/airs/" + files[i];
			}
		}
		return foundFile;
	}
	
	public static void main(String[] args) {
		  
		  try 
		  {   		
			  	NetcdfFileWriteable ncWrite = NetcdfFileWriteable.createNew(DIR + "/AIRS_AIRX3STM_Subset_GFDL.nc", false);
			  
			  	//Define dimensions and variables
			  	Dimension latDim = ncWrite.addDimension("lat", NUM_LAT);
			  	Dimension lonDim = ncWrite.addDimension("lon", NUM_LON);
			  	Dimension levDim = ncWrite.addDimension("lev", 12);
			  	Dimension monDim = ncWrite.addDimension("mon", NUM_MONTHS);
			  
			  	ArrayList dims3D = new ArrayList();
			  	dims3D.add(monDim);
			  	dims3D.add(latDim);
			  	dims3D.add(lonDim);
			  
			  	ArrayList dims4D = new ArrayList();
			  	dims4D.add(monDim);
			  	dims4D.add(levDim);
			  	dims4D.add(latDim);
			  	dims4D.add(lonDim);
			  
			  	ncWrite.addVariable("SurfSkinTemp_A", DataType.DOUBLE, dims3D);
				ncWrite.addVariable("SurfSkinTemp_D", DataType.DOUBLE, dims3D);
				ncWrite.addVariable("Temperature_A", DataType.DOUBLE, dims4D);
				ncWrite.addVariable("Temperature_D", DataType.DOUBLE, dims4D);
				ncWrite.addVariable("H2OVapMMR_A", DataType.DOUBLE, dims4D);
				ncWrite.addVariable("H2OVapMMR_D", DataType.DOUBLE, dims4D);
				ncWrite.addVariable("RelHumid_A", DataType.DOUBLE, dims4D);
				ncWrite.addVariable("RelHumid_D", DataType.DOUBLE, dims4D);

			  	ncWrite.create();
			  				  
				for (int m=0; m < NUM_MONTHS; m++)
				{
					System.out.print(m + " ");
					int year = 2002 + (m + 8) / 12;
					int month = (m + 8) % 12 + 1;
					String file = getFile(year, month);	
					
					NetcdfFile hdfFile = NetcdfFile.open(file);
					
					float[][] surfaceSkinTemp = ReadArray2D(hdfFile, "ascending/Data Fields/SurfSkinTemp_A");
					WriteArray2D(ncWrite, "SurfSkinTemp_A", surfaceSkinTemp, m);
					surfaceSkinTemp = ReadArray2D(hdfFile, "descending/Data Fields/SurfSkinTemp_D");
					WriteArray2D(ncWrite, "SurfSkinTemp_D", surfaceSkinTemp, m);
					
					float[][][] monthAirTemp = ReadArray3D(hdfFile, "ascending/Data Fields/Temperature_A");
					WriteArray3D(ncWrite, "Temperature_A", monthAirTemp, m);
					monthAirTemp = ReadArray3D(hdfFile, "descending/Data Fields/Temperature_D");
					WriteArray3D(ncWrite, "Temperature_D", monthAirTemp, m);
					
					float[][][] monthSpecHumid = ReadArray3D(hdfFile, "ascending/Data Fields/H2OVapMMR_A");
					WriteArray3D(ncWrite, "H2OVapMMR_A", monthSpecHumid, m);
					monthSpecHumid = ReadArray3D(hdfFile, "descending/Data Fields/H2OVapMMR_D");
					WriteArray3D(ncWrite, "H2OVapMMR_D", monthSpecHumid, m);
					
					float[][][] monthRelHumid = ReadArray3D(hdfFile, "ascending/Data Fields/RelHumid_A");
					WriteArray3D(ncWrite, "RelHumid_A", monthRelHumid, m);
					monthRelHumid = ReadArray3D(hdfFile, "descending/Data Fields/RelHumid_D");
					WriteArray3D(ncWrite, "RelHumid_D", monthRelHumid, m);
									
					hdfFile.close();
										
				}
				
				ncWrite.close();			
		  } 
		  catch (Exception ioe) 
		  {    
			  System.err.print("trying to open " + ioe.toString());
		  } 
		  finally 
		  { 
		  }
	}

}
