//Troy Masters 2012
//COHCDataProcessor: Calculates weighted average temperatures
//from Ocean Heat data as downloaded from http://www.nodc.noaa.gov/OC5/3M_HEAT_CONTENT/anomaly_data.html

import java.io.*;

public class COHCDataProcessor {
	
	//Directory that holds our .dat files from the tar gzip downloaded from NOAA
	public static String Directory = "C:\\Climate\\LevData\\";

	/**
	 * @param args
	 */
	public static void main(String[] args) 
	{
		//Used to weight each lat/lon gridcell based on surface area		
		double[][] globalWeights = new double[360][180];
		for (int lon=0; lon < 360; lon++)
		{
			for (int lat=0; lat < 180; lat++)
			{
				double lat1 = (90.0-(lat-1))*Math.PI/180.0;
				double lat2 = (90.0-(lat))*Math.PI/180.0;
				globalWeights[lon][lat] = (Math.PI/180*(Math.sin(lat1) - Math.sin(lat2))) / (4*Math.PI);
				//Surface Area = R^2(lon1-lon2)(sin (lat1) - sin(lat2))
			}
		}
		
		//Set up our measure depths (and the borders we'll use for weighting)
		int[] depthMeasures = {0,10,20,30,50,75,100,125,150,200,250,300,400,500,600,700,800}; 
		
		//Borders are half-way between the depth measure
		double[] depthBorders = new double[17];
		depthBorders[0] = 0;
		for (int dep=1; dep < depthBorders.length; dep++)
		{
			depthBorders[dep] = (depthMeasures[dep] + depthMeasures[dep-1])/2;
		}
		
		//Hold the 3-month globally averaged anomalies for each of the 16 layers
		double[][] tempAvgs = new double[(2011-1955)*4][16]; 
		
		//Percent of overall Earth surface area per layer
		double[] totalAreaWeights = new double[16];
		
		for (int year=1955; year <= 2010; year++) //Go through all full years
		{
			for (int monthStart=1; monthStart < 12; monthStart += 3) //Go through all full months
			{
				//Hideous code to get the filename of the associated year and month .dat file
				String file = "tan_";
				char tempYear1 = (char) ((year % 100 / 10) + 48);
				char tempYear2 = (char) ((year % 10) + 48);
				if (year >= 2010)
					tempYear1 = 'B';
				else if (year >= 2000)
					tempYear1 = 'A';
				file += tempYear1;
				file += tempYear2;
				file += tempYear1;
				file += tempYear2;
				Integer temp = monthStart;
				String tempString = temp.toString();
				if (tempString.length() < 2)
					tempString = '0' + tempString;
				file += tempString + "-";
				temp = monthStart+2;
				tempString = temp.toString();
				if (tempString.length() < 2)
					tempString = '0' + tempString;
				file += tempString + ".dat";
				file = Directory + file;
				
				//Read our data from this 3-month file into 3D array
				double[][][] fileData = new double[360][180][16];
				long n=0;
				try
			   	{
				   	 BufferedReader input =  new BufferedReader(new FileReader(file));
				     String line = "";
				     while ((line = input.readLine()) != null)
				     {
				    	 for (int i=0; i < 10; i++)
				    	 {
				    		 double val = Double.parseDouble(line.substring(8*i, 8*i+8));
				    		 int depth = (int) (n / (360*180));
				    		 int y = (int) ((n % (360*180)) / 360);
				    		 int x = (int) ((n % (360*180)) % 360);
				    		 fileData[x][y][depth] = val;
				    		 n++;
				    	 }
				      	 
				     }
			   	}
			   	catch (Exception e)
			   	{
			   	 System.err.println("Unable to read OHC Data: " + file + " " + e.getMessage());
			   	}
			   	
			   	//----------Perform averaging
			   	double[] weightedTempAnom = new double[16];
			   	for (int layer=0; layer < 16; layer++)
			   	{
			   		totalAreaWeights[layer] = 0;
			   		weightedTempAnom[layer] = 0;
			   	}
			   	
			   	//Go through each depth layer
			   	for (int layer=0; layer < 16; layer++)
			   	{
			   		for (int x=0; x < 360; x++)
			   		{
			   			for (int y=0; y < 180; y++)
			   			{
			   				if (fileData[x][y][layer] > -99)
			   				{
			   					totalAreaWeights[layer] += globalWeights[x][y];
			   					weightedTempAnom[layer] += globalWeights[x][y] * fileData[x][y][layer];
			   				}
			   			}
			   		}
			   		weightedTempAnom[layer] /= totalAreaWeights[layer]; //Divide by the percent of area actually used (~70%)
			   	}
			   	
			   	//Copy results into main array for this 3-month period
			   	for (int layer=0; layer < 16; layer++)
			   	{
			   		tempAvgs[(year-1955)*4+(monthStart/3)][layer] = weightedTempAnom[layer];
			   	}
			}
		}
		
	   				
		try  //Output results into "OHC_output.txt"
        {
	        Writer output = new OutputStreamWriter(new FileOutputStream(Directory + "OHC_output.txt"));
	        output.write("Temperature Anomalies By Layer\r\n");
	        //Here we assume all -99.99 refer to land or ocean floor
	        output.write("Approximate Weights By Volume Per Layer: \r\n");
	        for (int lay=0; lay < 16; lay++)
	        {
	        	output.write("\tLayer_" + depthMeasures[lay] + "\t" + totalAreaWeights[lay]*(depthBorders[lay+1]-depthBorders[lay]) + "\r\n");
	        }
	        
	        
	        output.write("\r\nYear");
	        for (int lay=0; lay < 16; lay++)
	        {
	        	output.write("\tLayer_" + depthMeasures[lay]);
	        }
	        output.write("\r\n");
	        for (int i=0; i < (2011-1955)*4; i++)
	        {
	        	double year = (i)/4.0 + 1955;
	        	output.write(year + "");
	        	for (int lay=0; lay < 16; lay++)
		        {
		        	output.write("\t" + tempAvgs[i][lay]);
		        }
	        	output.write("\r\n");
	        }
	        
	        
	        output.close();
	       
        }
        catch (Exception e)
        {
        	System.err.println("Unable to output file: " + e.getMessage());
        }
	}
}





