#Author: Troy Masters
#This script calculates the flux contribution for clear and all-sky kernels
#from AIRS data and the GFDL radiative kernels

#NOTE: to run this script you must first process the AIRS data to match up with the resolution below,
#along with the Soden radiative kernels

#set your working directory
setwd("C:/Climate/Masters_2012/extra")

#must first install ncdf
library(ncdf)

#AIRS horizontal coordinates go -90 N to 90N, -180 to 180.
AIRS.nc<-open.ncdf("AIRS_AIRX3STM_Subset_GFDL.nc") #open our nc file
#pressure level 1000 to 100

#kernel pressure level18=1050, 17=1000, 850, etc.
#units for layers are per 100 hPa!
kernel_atmt.nc<-open.ncdf("lw_t_gfdl_std.nc")
kernel_lwwv.nc<-open.ncdf("lw_q_gfdl_std.nc")
kernel_swwv.nc<-open.ncdf("sw_q_gfdl_std.nc")
kernel_atmt_clr.nc<-open.ncdf("lwclr_t_gfdl_std.nc")
kernel_lwwv_clr.nc<-open.ncdf("lwclr_q_gfdl_std.nc")
kernel_swwv_clr.nc<-open.ncdf("swclr_q_gfdl_std.nc")

numLon<-144
numLat<-90

#Insert global weight and global avg ts functions here
############determine weight for each horizontal gridcell based on surface area####
globalWeight<-array(0,dim=c(numLon, numLat))
for (lat in 1:numLat)
{
	lat1<-(90-(lat-1)*(180/numLat))*pi/180
	lat2<-(90-(lat)*(180/numLat))*pi/180
	globalWeight[,lat]<-(360/numLon)*pi/180*(sin(lat1) - sin(lat2))
	#Surface Area = R^2(lon1-lon2)(sin (lat1) - sin(lat2))
}
globalWeight<-globalWeight/(4*pi)

##############Function that calculate weighted global average from gridded data####
getGlobalAverageTs<-function(data, months)
{
	globalVar<-numeric(months)
	for (month in 1:(months))
	{
		globalVar[month]<-sum(data[,,month]*globalWeight)
	}
	ts(globalVar, start=c(2002,9), freq=12)
}


###Function to calculate flux contributions per grid point for 2D kernels############
calcSurfaceGriddedFluxContributions<-function(kernelNC, kernelVar, airsNC, airsVar, numMonths)
{
	fluxContributions<-array(0, dim=c(numLon, numLat, numMonths))
	airsValues<-array(0, dim=c(numLon, numLat, numMonths))
	airsAverages<-array(0, dim=c(numLon,numLat,12))
	
	for (month in 1:numMonths)
	{
		#must shift from -180:180 to 0:360 to match kernels
		airsValues[,,month]<-(get.var.ncdf(airsNC, paste(airsVar,"_A",sep=""), start=c(1,1,month), count=c(numLon,numLat,1)) +
				      get.var.ncdf(airsNC, paste(airsVar,"_D",sep=""), start=c(1,1,month), count=c(numLon,numLat,1)))/2
		
				
		if (month > 4 && month <= 100) #only baseline for full years (start in 2003)
		{ 
			m_ind<-((month+8-1) %% 12 + 1)
			airsAverages[,,m_ind]<-airsAverages[,,m_ind] + airsValues[,,month]/8
		}
	}
	
	for (month in 1:numMonths)
	{
		monIndex<-((month+8-1) %% 12 + 1)
		kernel<-get.var.ncdf(kernelNC, kernelVar, start=c(1,1,18,monIndex), count=c(numLon, numLat,1,1))
		airs.anom<-airsValues[,,month]-airsAverages[,,monIndex]
		airs.anom<-rbind(airs.anom[(numLon/2+1):numLon,], airs.anom[1:(numLon/2),]) #shift from -180:180 to 0:360 to match kernel
		fluxContributions[,,month]<-airs.anom*kernel
	}
	fluxContributions
}


#######Here we actually get the flux contributions for clr and all-sky albedo and surface temp fluxes
surfaceTempFluxes<-calcSurfaceGriddedFluxContributions(kernel_atmt.nc, "lw_t", AIRS.nc, "SurfSkinTemp", 110)
surfaceTemp_clrFluxes<-calcSurfaceGriddedFluxContributions(kernel_atmt_clr.nc, "lwclr_t", AIRS.nc, "SurfSkinTemp", 110)

surfaceFlux.ts<-getGlobalAverageTs(surfaceTempFluxes,110)
surface_clrFlux.ts<-getGlobalAverageTs(surfaceTemp_clrFluxes,110)

#####Prepare for 3D kernels by setting up layerBorders and thickness (units are in hPa)################
layerBorders<-numeric(13)
layerBorders[1]<-85
layerBorders[2]<-125
layerBorders[3]<-175
layerBorders[4]<-225
layerBorders[5]<-275
layerBorders[6]<-350
layerBorders[7]<-450
layerBorders[8]<-550
layerBorders[9]<-650
layerBorders[10]<-775
layerBorders[11]<-887.5
layerBorders[12]<-962.5
layerBorders[13]<-1050

pressureLayers<-numeric(12)
pressureLayers[1]<-100
pressureLayers[2]<-150
pressureLayers[3]<-200
pressureLayers[4]<-250
pressureLayers[5]<-300
pressureLayers[6]<-400
pressureLayers[7]<-500
pressureLayers[8]<-600
pressureLayers[9]<-700
pressureLayers[10]<-850
pressureLayers[11]<-925
pressureLayers[12]<-1000


layerThickness<-numeric(12)
for (layer in 1:12)
{
	layerThickness[layer]<-layerBorders[layer+1]-layerBorders[layer]
}


###Function to calculate flux contributions per horizontal grid point for atmospheric temperature kernels##
##(Don't have to worry about standard anomaly like we do for water vapor calcs)###########################
calcAtmoTempGriddedFluxContributions<-function(kernelNC, kernelVar, airsNC, airsVar, numMonths)
{
	fluxContributions<-array(0, dim=c(numLon,numLat, numMonths))
	airsValues<-array(0, dim=c(numLon,numLat,12,numMonths))
	airsAverages<-array(0, dim=c(numLon,numLat,12,numMonths))
	
	#grab values and averages from AIRS that we can calc anomalies from later
	for (month in 1:numMonths)
	{
		airsValues[(numLon/2+1):numLon,,,month]<-(get.var.ncdf(airsNC, paste(airsVar,"_A",sep=""), start=c(1,1,1,month), count=c((numLon/2),numLat,12,1)) +
				      get.var.ncdf(airsNC, paste(airsVar,"_D",sep=""), start=c(1,1,1,month), count=c((numLon/2),numLat,12,1)))/2
		airsValues[1:(numLon/2),,,month]<-(get.var.ncdf(airsNC, paste(airsVar,"_A",sep=""), start=c(65,1,1,month), count=c((numLon/2),numLat,12,1)) +
				      get.var.ncdf(airsNC, paste(airsVar,"_D",sep=""), start=c(65,1,1,month), count=c((numLon/2),numLat,12,1)))/2
				
		if (month > 4 && month <= 100) #only baseline for full years (start in 2003)
		{ 
			m_ind<-((month+8-1) %% 12 + 1)
			airsAverages[,,,m_ind]<-airsAverages[,,,m_ind] + airsValues[,,,month]/8
		}
	}

	for (month in 1:numMonths)
	{
		monIndex<-((month+8-1) %% 12 + 1)
		#here we skip right to the "100" level, which is why we add 5
		kernel<-get.var.ncdf(kernelNC, kernelVar, start=c(1,1,6,monIndex), count=c(numLon,numLat,12,1))
		airs.anom<-airsValues[,,,month]-airsAverages[,,,monIndex]
		#layers are inverted for AIRS
		fluxAllLevels<-airs.anom[,,12:1]*kernel*layerThickness/100 #divide by 100hPA
		
		for (y in 1:numLat)  #tropopause varies, we only want to sum up levels of troposphere
		{
			lat<-(y-.5)*(180/numLat) - 90
			tropoPause<-100+abs(lat)/90*200
			for (layer in 1:12)
			{
				weight<-0
				if (layerBorders[layer] > tropoPause)
				{	#full layer should be included
					weight<-1.0
				}
				else if (layerBorders[layer+1] < tropoPause)
				{	#none of this layer should be included
					weight<-0
				}
				else
				{
					#part of this layer should be included
					weight<-(layerBorders[layer+1]-tropoPause)/(layerThickness[layer])
				}
				fluxContributions[,y,month]<-fluxContributions[,y,month]+(fluxAllLevels[,y,layer]*weight)
			}
		}
	}
	fluxContributions
}


#########Actually get all and clr-sky atmospheric temperature flux contributions 
atmoTempFluxes<-calcAtmoTempGriddedFluxContributions(kernel_atmt.nc,"lw_t", AIRS.nc, "Temperature", 110)
atmoTemp_clrFluxes<-calcAtmoTempGriddedFluxContributions(kernel_atmt_clr.nc,"lwclr_t", AIRS.nc, "Temperature", 110)

atmoTempFluxes.ts<-getGlobalAverageTs(atmoTempFluxes,110)
atmoTemp_clrFluxes.ts<-getGlobalAverageTs(atmoTemp_clrFluxes,110)

###Function to calculate saturation pressure
#Takes T in K, returns pressure in hPa
satPressure<-function(T)
{
	#exp(77.3450+.0057*T-7235/T)/t^8.2 #K to Pa
	#6.22*exp(17.62*T/(243.12 + T)) #C to hPa
	10^ (  -7.90298*(373.16/T-1) #K to hPa                                                             [1] 
             +5.02808* log10(373.16/T) 
             -1.3816* 10^(-7) *(10^(11.344*(1-T/373.16))-1) 
             + 8.1328*10^(-3)* (10^(-3.49149*(373.16/T-1))-1) 
             + log10(1013.246)  ) 

}

#T in K, P in hPA, RH between 0 and 1
#returns kg/kg
specificHumidity<-function(RH, T, P)
{
	e_s<-satPressure(T) #sat pressure in hPa
	e<-e_s*RH
	eps<-0.622
	q<-(eps*e/(P-(1-eps)*e))
}


####Get the standard anomaly (change in q based on 1 degree change in T with constant RH)
###for each 3D coordinate per month of the year
q.standardAnomaly<-array(0, dim=c(numLon, numLat, 12, 12))
for (month in 5:28)
{
	m_ind<-((month+8-1) %% 12 + 1)
	for (level in 1:12)
	{
		airs_rh<-array(0, dim=c(numLon,numLat))
		airs_rh[(numLon/2+1):numLon,]<-(get.var.ncdf(AIRS.nc, "RelHumid_A", start=c(1,1,level,month), count=c((numLon/2),numLat,1,1)) +
				      get.var.ncdf(AIRS.nc, "RelHumid_D", start=c(1,1,level,month), count=c((numLon/2),numLat,1,1)))/2
		airs_rh[1:(numLon/2),]<-(get.var.ncdf(AIRS.nc, "RelHumid_A", start=c(65,1,level,month), count=c((numLon/2),numLat,1,1)) +
				      get.var.ncdf(AIRS.nc, "RelHumid_D", start=c(65,1,level,month), count=c((numLon/2),numLat,1,1)))/2
		airs_rh<-airs_rh/100 #convert from % to decimal
		
		airs_T<-array(0, dim=c(numLon,numLat))
		airs_T[(numLon/2+1):numLon,]<-(get.var.ncdf(AIRS.nc, "Temperature_A", start=c(1,1,level,month), count=c((numLon/2),numLat,1,1)) +
				      get.var.ncdf(AIRS.nc, "Temperature_D", start=c(1,1,level,month), count=c((numLon/2),numLat,1,1)))/2
		airs_T[1:(numLon/2),]<-(get.var.ncdf(AIRS.nc, "Temperature_A", start=c(65,1,level,month), count=c((numLon/2),numLat,1,1)) +
				      get.var.ncdf(AIRS.nc, "Temperature_D", start=c(65,1,level,month), count=c((numLon/2),numLat,1,1)))/2
		for (x in 1:numLon)
		{
			for (y in 1:numLat)
			{
				logPerturbed<-log(specificHumidity(airs_rh[x,y], airs_T[x,y]+1.0,1000))
				logNormal<-log(specificHumidity(airs_rh[x,y], airs_T[x,y],1000))
				logDiff<-logPerturbed-logNormal
				if (is.na(logDiff)) logDiff<-9999 #high number so flux contribution will be nothing
				q.standardAnomaly[x,y,level,m_ind]<-q.standardAnomaly[x,y,level,m_ind] + (logDiff)/2 #2 year average
			}
		}
	}
}

###Function to calculate flux contributions per horizontal grid point for 3D water vapor kernels##
calcWaterVaporGriddedFluxContributions<-function(kernelNC, kernelVar, airsNC, airsVar, numMonths)
{
	fluxContributions<-array(0, dim=c(numLon,numLat, numMonths))
	airsValues<-array(0, dim=c(numLon,numLat,12,numMonths))
	airsAverages<-array(0, dim=c(numLon,numLat,12,numMonths))
	
	#grab values and averages from AIRS that we can calc anomalies from later
	for (month in 1:numMonths)
	{
		airsValues[(numLon/2+1):numLon,,,month]<-(get.var.ncdf(airsNC, paste(airsVar,"_A",sep=""), start=c(1,1,1,month), count=c((numLon/2),numLat,12,1)) +
				      get.var.ncdf(airsNC, paste(airsVar,"_D",sep=""), start=c(1,1,1,month), count=c((numLon/2),numLat,12,1)))/2
		airsValues[1:(numLon/2),,,month]<-(get.var.ncdf(airsNC, paste(airsVar,"_A",sep=""), start=c(65,1,1,month), count=c((numLon/2),numLat,12,1)) +
				      get.var.ncdf(airsNC, paste(airsVar,"_D",sep=""), start=c(65,1,1,month), count=c((numLon/2),numLat,12,1)))/2
				
		airsValues[,,,month]<-airsValues[,,,month]/1000 #go from g/kg to kg/kg

		if (month > 4 && month <= 100) #only baseline for full years (start in 2003)
		{ 
			m_ind<-((month+8-1) %% 12 + 1)
			airsAverages[,,,m_ind]<-airsAverages[,,,m_ind] + airsValues[,,,month]/8
		}
	}

	for (month in 1:numMonths)
	{
		monIndex<-((month+8-1) %% 12 + 1)

		#here we skip right to the "100" level
		kernel<-get.var.ncdf(kernelNC, kernelVar, start=c(1,1,6,monIndex), count=c(numLon,numLat,12,1))
		airs.anom<-log(airsValues[,,,month])-log(airsAverages[,,,monIndex])
		
		#layers are inverted for AIRS
		fluxAllLevels<-airs.anom[,,12:1]*kernel/q.standardAnomaly[,,12:1,monIndex]*layerThickness/100 #divide by 100hPA
	
		for (y in 1:numLat)
		{
			lat<-(y-.5)*(180/numLat) - 90
			tropoPause<-100+abs(lat)/90*200
			for (layer in 1:12)
			{
				weight<-0
				if (layerBorders[layer] > tropoPause)
				{	#full layer should be included
					weight<-1.0
				}
				else if (layerBorders[layer+1] < tropoPause)
				{	#none of this layer should be included
					weight<-0
				}
				else
				{
					#part of this layer should be included
					weight<-(layerBorders[layer+1]-tropoPause)/(layerThickness[layer])
				}
				fluxContributions[,y,month]<-fluxContributions[,y,month]+(fluxAllLevels[,y,layer]*weight)
			}
		}
	}
	fluxContributions
}


########Actually get flux contributions for clear and all-sky sw and lw water vapor kernels
lwWvFluxes<-calcWaterVaporGriddedFluxContributions(kernel_lwwv.nc,"lw_q", AIRS.nc, "H2OVapMMR", 110)
swWvFluxes<-calcWaterVaporGriddedFluxContributions(kernel_swwv.nc,"sw_q", AIRS.nc, "H2OVapMMR", 110)
lwWv_clrFluxes<-calcWaterVaporGriddedFluxContributions(kernel_lwwv_clr.nc,"lwclr_q", AIRS.nc, "H2OVapMMR", 110)
swWv_clrFluxes<-calcWaterVaporGriddedFluxContributions(kernel_swwv_clr.nc,"swclr_q", AIRS.nc, "H2OVapMMR", 110)

lwWvFluxes.ts<-getGlobalAverageTs(lwWvFluxes,110)
swWvFluxes.ts<-getGlobalAverageTs(swWvFluxes,110)
lwWv_clrFluxes.ts<-getGlobalAverageTs(lwWv_clrFluxes,110)
swWv_clrFluxes.ts<-getGlobalAverageTs(swWv_clrFluxes,110)

data<-ts.union(surfaceFlux.ts, atmoTempFluxes.ts, lwWvFluxes.ts, swWvFluxes.ts,
	surface_clrFlux.ts, atmoTemp_clrFluxes.ts, lwWv_clrFluxes.ts, swWv_clrFluxes.ts)

#Output the data
write.table(data, "AIRS_GFDL_GlobalFluxContributions_2002_9-2011_10.txt")

