#Troy Masters 2012

#set working directory here
setwd("C:/Climate/Masters_2012")

library(ncdf) #for reading netCDF files

#end of period (do full length of CERES available obs)
end<-c(2011,6)

#########################
#HELPER FUNCTIONS
############################
#Calculate the anomaly based on the yearly frequency
anom<-function(x, freq=12)
{
	totalSum<-numeric(freq)
	numObs<-numeric(freq)
	for (mon in 1:length(x))
	{
		totalSum[mon %% freq + 1]<-totalSum[mon %% freq + 1] + x[mon]
		numObs[mon %% freq + 1]<-numObs[mon %% freq + 1] + 1
	}
	anom<-numeric(length(x))
	for (mon in 1:length(x))
	{
		anom[mon]<-x[mon]-(totalSum[mon %% freq + 1]/numObs[mon %% freq + 1])	
	}	
	anom<-ts(anom,start=tsp(x)[1],freq=freq) 
}

#get nicely formatted row of data based on regression of CRF against temperatures
##################################
getRegressionRow<-function(fluxAll, fluxSw, fluxLw, temps, monthsAgg)
{
	freq<-12/monthsAgg

        fluxAll.agg<-anom(aggregate(fluxAll, nfrequency=freq)/monthsAgg, freq)
	fluxSw.agg<-anom(aggregate(fluxSw, nfrequency=freq)/monthsAgg, freq)
	fluxLw.agg<-anom(aggregate(fluxLw, nfrequency=freq)/monthsAgg, freq)
	temps.agg<-anom(aggregate(temps, nfrequency=freq)/monthsAgg, freq)	

	regNet.lm<-lm(fluxAll.agg ~ temps.agg, data=ts.union(fluxAll.agg, temps.agg))
	regSw.lm<-lm(fluxSw.agg ~ temps.agg, data=ts.union(fluxSw.agg, temps.agg))
	regLw.lm<-lm(fluxLw.agg ~ temps.agg, data=ts.union(fluxLw.agg, temps.agg))
	
	#output our R^2 values to console
	print(summary(regNet.lm)$r.squared)
	print(summary(regSw.lm)$r.squared)
	print(summary(regLw.lm)$r.squared)

	y1<-paste(format(round(coef(regNet.lm)[2],2), nsmall=2), "", format(round(coef(regNet.lm)[2]-confint(regNet.lm)[2,1], 2), nsmall=2)) 
	y2<-paste(format(round(coef(regSw.lm)[2],2), nsmall=2), "", format(round(coef(regSw.lm)[2]-confint(regSw.lm)[2,1], 2), nsmall=2))
	y3<-paste(format(round(coef(regLw.lm)[2],2), nsmall=2), "", format(round(coef(regLw.lm)[2]-confint(regLw.lm)[2,1], 2), nsmall=2))
	regRow<-c(y1, y2, y3)
}

getRegressionRowLw<-function(fluxLw, temps, monthsAgg)
{
        fluxLw.agg<-anom(aggregate(fluxLw, nfrequency=12/monthsAgg)/monthsAgg, 12/monthsAgg)
	temps.agg<-anom(aggregate(temps, nfrequency=12/monthsAgg)/monthsAgg, 12/monthsAgg)	
	regLw.lm<-lm(fluxLw.agg ~ temps.agg, data=ts.union(fluxLw.agg, temps.agg))

	#output our R^2 values to console
	print(summary(regLw.lm)$r.squared)

	y3<-paste(format(round(coef(regLw.lm)[2],2), nsmall=2), "", format(round(coef(regLw.lm)[2]-confint(regLw.lm)[2,1], 2), nsmall=2))
	regRow<-c(NA, NA, y3)
}

#doKernelAdjust - perform kernel adjustments
#crf.anom - CRF time series of anomalies
#useAirs - TRUE use AIRS (only starts 9-2002) for LW, FALSE user ERA-Interim LW adjusts
#lwAdjust - TRUE perform adjustments to OLR
#swAdjust - TRUE perform adjustments to SW 
#returns - CRF adjusted to become R_Cloud
doKernelAdjust<-function(crf.anom, useAirs, lwAdjust, swAdjust)
{
	rcloud.anom<-crf.anom
	era_fluxes<-read.table("ERA_GFDL_GlobalFluxContributions_2000_1-2011_11.txt")
	airs_fluxes<-read.table("AIRS_GFDL_GlobalFluxContributions_2002_9-2011_10.txt")
	if (!useAirs) #need full period adjustments
	{
		surfaceTempFluxAdj<-ts(era_fluxes[[2]]-era_fluxes[[7]], start=c(2000,1), freq=12)
		atmoTempFluxAdj<-ts(era_fluxes[[3]]-era_fluxes[[8]], start=c(2000,1), freq=12)
		lwWvFluxAdj<-ts(era_fluxes[[4]]-era_fluxes[[9]], start=c(2000,1), freq=12)
		swWvFluxAdj<-ts(era_fluxes[[10]]-era_fluxes[[5]], start=c(2000,1), freq=12)
	}
	if (useAirs) #only need to adjust over AIRS period
	{
		surfaceTempFluxAdj<-ts(airs_fluxes[[1]]-airs_fluxes[[5]], start=c(2002,9), freq=12)
		atmoTempFluxAdj<-ts(airs_fluxes[[2]]-airs_fluxes[[6]], start=c(2002,9), freq=12)
		lwWvFluxAdj<-ts(airs_fluxes[[3]]-airs_fluxes[[7]], start=c(2002,9), freq=12)
		swWvFluxAdj<-ts(airs_fluxes[[8]]-airs_fluxes[[4]], start=c(2002,9), freq=12)
	}	

	albedoFluxAdj<-ts(era_fluxes[[1]]-era_fluxes[[6]], start=c(2000,1), freq=12)
	CO2Adjust<--1*ts((1:140)/140*0.25 * 0.16, start=c(2000,3), freq=12)
	if (lwAdjust)
	{
		rcloud.anom<-rcloud.anom + surfaceTempFluxAdj +atmoTempFluxAdj + CO2Adjust + lwWvFluxAdj
	}
	if (swAdjust)
	{
		rcloud.anom<-rcloud.anom + swWvFluxAdj - albedoFluxAdj
	}	
	rcloud.anom	
}

#Main function to perform work
#tempFiles - list of temperature file names to use
#tempStartDates - dates of first temperature in the corresponding temperature files
#endDates - last date to use for the coressponding temperature file in the analysis
createAnalysisTables<-function(tempFiles, tempStartDates, endDates)
{

###########################################
#1. PERFORM ANALYSIS OVER TERRA PERIOD
###########################################
	start<-c(2000,3)

	terraStart<-c(2000,3)

	#Read TERRA clear and all-sky
	terra.nc<-open.ncdf("CERES_SSF1deg-Month-lite_Terra_Ed2.6_Subset_200003-201106.nc")
	terra_net_all<-ts(get.var.ncdf(terra.nc, "gtoa_net_all_mon"), start=terraStart, freq=12)
	terra_sw_all<-ts(get.var.ncdf(terra.nc, "gtoa_sw_all_mon"), start=terraStart, freq=12)
	terra_sw_clr<-ts(get.var.ncdf(terra.nc, "gtoa_sw_clr_mon"), start=terraStart, freq=12)
	terra_lw_all<-ts(get.var.ncdf(terra.nc, "gtoa_lw_all_mon"), start=terraStart, freq=12)
	terra_lw_clr<-ts(get.var.ncdf(terra.nc, "gtoa_lw_clr_mon"), start=terraStart, freq=12)
	close.ncdf(terra.nc)

	#Read EBAF clear-sky
	ebaf.nc<-open.ncdf("CERES_EBAF-TOA_Ed2.6r_Subset_200003-201106.nc")
	ebaf_net_clr<-ts(get.var.ncdf(ebaf.nc, "gtoa_net_clr_mon"), start=terraStart, freq=12)
	ebaf_sw_clr<-ts(get.var.ncdf(ebaf.nc, "gtoa_sw_clr_mon"), start=terraStart, freq=12)
	ebaf_lw_clr<-ts(get.var.ncdf(ebaf.nc, "gtoa_lw_clr_mon"), start=terraStart, freq=12)
	close.ncdf(ebaf.nc)

	#get ERA clear-sky data
	eraData<-read.table("era_flux_data.txt", header=TRUE)
	era_sw_inc<-ts(eraData["sw_inc"], freq=12, start=2000)
	era_sw_clr<-ts(eraData["sw_out_clr"], freq=12, start=2000)
	era_lw_clr<-ts(eraData["lw_out_clr"], freq=12, start=2000)

	#get MERRA clear-sky data
	merraData<-read.table("merra_flux_data.txt")
	merra_sw_clr<-ts(merraData[[2]], freq=12, start=2000)
	merra_lw_clr<-ts(merraData[[3]], freq=12, start=2000)

	####Calculate CRF values
	#First for TERRA
	terra_crf_net<-window((terra_sw_clr+terra_lw_clr)-(terra_sw_all+terra_lw_all), start=start, end=end)
	terra_crf_net.anom<-anom(terra_crf_net)
	terra_crf_sw.anom<-anom(window(terra_sw_clr-terra_sw_all, start=start, end=end),12)
	terra_crf_lw.anom<-anom(window(terra_lw_clr-terra_lw_all, start=start, end=end),12)

	#Then for ERA-Interim
	####combine just sw and lw to avoid incident solar inconsistency
	era_crf_net<-window((era_sw_clr+era_lw_clr)-(terra_sw_all+terra_lw_all), start=start, end=end)
	era_crf_net.anom<-anom(era_crf_net, 12)
	era_crf_sw.anom<-anom(window(era_sw_clr-terra_sw_all, start=start, end=end),12)
	era_crf_lw.anom<-anom(window(era_lw_clr-terra_lw_all, start=start, end=end),12)

	#Then for EBAF
	ebaf_crf_net<-window((ebaf_sw_clr+ebaf_lw_clr)-(terra_sw_all+terra_lw_all), start=start, end=end)
	ebaf_crf_net.anom<-anom(ebaf_crf_net, 12)
	ebaf_crf_sw.anom<-anom(window(ebaf_sw_clr-terra_sw_all, start=start, end=end),12)
	ebaf_crf_lw.anom<-anom(window(ebaf_lw_clr-terra_lw_all, start=start, end=end),12)

	#Then for MERRA
	merra_crf_net<-window((merra_sw_clr+merra_lw_clr)-(terra_sw_all+terra_lw_all), start=start, end=end)
	merra_crf_net.anom<-anom(merra_crf_net, 12)
	merra_crf_sw.anom<-anom(window(merra_sw_clr-terra_sw_all, start=start, end=end),12)
	merra_crf_lw.anom<-anom(window(merra_lw_clr-terra_lw_all, start=start, end=end),12)

	terraMatrix<-array(0, dim=c(length(tempFiles)*4, 3))

for (i in 1:length(tempFiles))
{

	temp.ts<-ts(read.table(tempFiles[i])[[2]], start=tempStartDates[[i]], freq=12)
	temp.anom<-anom(window(temp.ts, start=start, end=endDates[[i]]))

	#First do them for CRF (unadjusted regressions)
	CERES.monthly<-getRegressionRow(window(terra_crf_net.anom, end=endDates[[i]]), window(terra_crf_sw.anom, end=endDates[[i]]), window(terra_crf_lw.anom, end=endDates[[i]]), temp.anom, 1)
	ERA.monthly<-getRegressionRow(window(era_crf_net.anom, end=endDates[[i]]), window(era_crf_sw.anom, end=endDates[[i]]), window(era_crf_lw.anom, end=endDates[[i]]), temp.anom, 1)
	EBAF.monthly<-getRegressionRow(window(ebaf_crf_net.anom, end=endDates[[i]]), window(ebaf_crf_sw.anom, end=endDates[[i]]), window(ebaf_crf_lw.anom, end=endDates[[i]]), temp.anom, 1)
	MERRA.monthly<-getRegressionRow(window(merra_crf_net.anom, end=endDates[[i]]), window(merra_crf_sw.anom, end=endDates[[i]]), window(merra_crf_lw.anom, end=endDates[[i]]), temp.anom, 1)

	tblMatrix1<-rbind(CERES.monthly, ERA.monthly, EBAF.monthly, MERRA.monthly)

	#Perform kernel adjustments to go from CRF values to R_cld values
	terra_rcloud_net.anom<-anom(window(doKernelAdjust(terra_crf_net.anom, FALSE, TRUE, TRUE), end=endDates[[i]])) 
	terra_rcloud_sw.anom<-anom(window(doKernelAdjust(terra_crf_sw.anom, FALSE, FALSE, TRUE), end=endDates[[i]])) 
	terra_rcloud_lw.anom<-anom(window(doKernelAdjust(terra_crf_lw.anom, FALSE, TRUE, FALSE), end=endDates[[i]])) 
	ebaf_rcloud_net.anom<-anom(window(doKernelAdjust(ebaf_crf_net.anom, FALSE, TRUE, TRUE), end=endDates[[i]])) 
	ebaf_rcloud_sw.anom<-anom(window(doKernelAdjust(ebaf_crf_sw.anom, FALSE, FALSE, TRUE), end=endDates[[i]])) 
	ebaf_rcloud_lw.anom<-anom(window(doKernelAdjust(ebaf_crf_lw.anom, FALSE, TRUE, FALSE), end=endDates[[i]])) 
	era_rcloud_net.anom<-anom(window(doKernelAdjust(era_crf_net.anom, FALSE, TRUE, TRUE), end=endDates[[i]])) 
	era_rcloud_sw.anom<-anom(window(doKernelAdjust(era_crf_sw.anom, FALSE, FALSE, TRUE), end=endDates[[i]])) 
	era_rcloud_lw.anom<-anom(window(doKernelAdjust(era_crf_lw.anom, FALSE, TRUE, FALSE), end=endDates[[i]])) 
	merra_rcloud_net.anom<-anom(window(doKernelAdjust(merra_crf_net.anom, FALSE, TRUE, TRUE), end=endDates[[i]])) 
	merra_rcloud_sw.anom<-anom(window(doKernelAdjust(merra_crf_sw.anom, FALSE, FALSE, TRUE), end=endDates[[i]])) 
	merra_rcloud_lw.anom<-anom(window(doKernelAdjust(merra_crf_lw.anom, FALSE, TRUE, FALSE), end=endDates[[i]]))
	
	CERES.monthly<-getRegressionRow(terra_rcloud_net.anom, terra_rcloud_sw.anom, terra_rcloud_lw.anom, temp.anom, 1)
	ERA.monthly<-getRegressionRow(era_rcloud_net.anom, era_rcloud_sw.anom, era_rcloud_lw.anom, temp.anom, 1)
	EBAF.monthly<-getRegressionRow(ebaf_rcloud_net.anom, ebaf_rcloud_sw.anom, ebaf_rcloud_lw.anom, temp.anom, 1)
	MERRA.monthly<-getRegressionRow(merra_rcloud_net.anom, merra_rcloud_sw.anom, merra_rcloud_lw.anom, temp.anom, 1)

	tblMatrix2<-rbind(CERES.monthly, ERA.monthly, EBAF.monthly, MERRA.monthly) 
	
	#here we merge the two regression tables (R_cld vs. Temp and CRF vs. Temp) with (CRF vs. Temp in parenthesis)
	tblMatrix<-array(0, dim=c(nrow(tblMatrix1), ncol(tblMatrix1)))
	for (row in 1:nrow(tblMatrix1))
	{
		for (col in 1:ncol(tblMatrix1))
		{
			tblMatrix[row,col]<-paste(tblMatrix2[row,col], "  (", tblMatrix1[row,col], ")", sep='')
		}
	}

	terraMatrix[((i-1)*4+1):(i*4), 1:3]<-tblMatrix
}

##output results	
df<-data.frame(terraMatrix)
for (i in 1:length(tempFiles))
{
	rownames(df)[((i-1)*4+1)]<-paste("CERES and", tempFiles[i])
	rownames(df)[((i-1)*4+2)]<-paste("ERA and", tempFiles[i])
	rownames(df)[((i-1)*4+3)]<-paste("EBAF and", tempFiles[i])
	rownames(df)[((i-1)*4+4)]<-paste("MERRA and", tempFiles[i])
}
write.table(df, file="CloudFeedback_TERRA_temps.txt",sep="\t")

###########################################
#2. PERFORM ANALYSIS OVER AQUA PERIOD
###########################################
start<-c(2002,9)
aquaStart<-c(2002,7)

aqua.nc<-open.ncdf("CERES_SSF1deg-Month-lite_Aqua_Ed2.6_Subset_200207-201106.nc") #open our nc file
aqua_sw_all<-ts(get.var.ncdf(aqua.nc, "gtoa_sw_all_mon"), start=aquaStart, freq=12)
aqua_sw_clr<-ts(get.var.ncdf(aqua.nc, "gtoa_sw_clr_mon"), start=aquaStart, freq=12)
aqua_lw_all<-ts(get.var.ncdf(aqua.nc, "gtoa_lw_all_mon"), start=aquaStart, freq=12)
aqua_lw_clr<-ts(get.var.ncdf(aqua.nc, "gtoa_lw_clr_mon"), start=aquaStart, freq=12)
close.ncdf(aqua.nc)

#here we average Terra and Aqua satellite observations
ceres_sw_all<-(aqua_sw_all+window(terra_sw_all, start=aquaStart))/2
ceres_sw_clr<-(aqua_sw_clr+window(terra_sw_clr, start=aquaStart))/2
ceres_lw_all<-(aqua_lw_all+window(terra_lw_all, start=aquaStart))/2
ceres_lw_clr<-(aqua_lw_clr+window(terra_lw_clr, start=aquaStart))/2


####Calculate CRF values
ceres_crf_net<-window((ceres_sw_clr+ceres_lw_clr)-(ceres_sw_all+ceres_lw_all), start=start, end=end)
ceres_crf_net.anom<-anom(ceres_crf_net)
ceres_crf_sw.anom<-anom(window(ceres_sw_clr-ceres_sw_all, start=start, end=end),12)
ceres_crf_lw.anom<-anom(window(ceres_lw_clr-ceres_lw_all, start=start, end=end),12)

terra_crf_net<-window((terra_sw_clr+terra_lw_clr)-(ceres_sw_all+ceres_lw_all), start=start, end=end)
terra_crf_net.anom<-anom(terra_crf_net)
terra_crf_sw.anom<-anom(window(terra_sw_clr-ceres_sw_all, start=start, end=end),12)
terra_crf_lw.anom<-anom(window(terra_lw_clr-ceres_lw_all, start=start, end=end),12)

aqua_crf_net<-window((aqua_sw_clr+aqua_lw_clr)-(ceres_sw_all+ceres_lw_all), start=start, end=end)
aqua_crf_net.anom<-anom(aqua_crf_net)
aqua_crf_sw.anom<-anom(window(aqua_sw_clr-ceres_sw_all, start=start, end=end),12)
aqua_crf_lw.anom<-anom(window(aqua_lw_clr-ceres_lw_all, start=start, end=end),12)


##re-calculate other anomalies
era_crf_net<-window((era_sw_clr+era_lw_clr)-(ceres_sw_all+ceres_lw_all), start=start, end=end)
era_crf_net.anom<-anom(era_crf_net, 12)
era_crf_sw.anom<-anom(window(era_sw_clr-ceres_sw_all, start=start, end=end),12)
era_crf_lw.anom<-anom(window(era_lw_clr-ceres_lw_all, start=start, end=end),12)

#Then for EBAF
ebaf_crf_net<-window((ebaf_sw_clr+ebaf_lw_clr)-(ceres_sw_all+ceres_lw_all), start=start, end=end)
ebaf_crf_net.anom<-anom(ebaf_crf_net, 12)
ebaf_crf_sw.anom<-anom(window(ebaf_sw_clr-ceres_sw_all, start=start, end=end),12)
ebaf_crf_lw.anom<-anom(window(ebaf_lw_clr-ceres_lw_all, start=start, end=end),12)

#and for MERRA
merra_crf_net<-window((merra_sw_clr+merra_lw_clr)-(ceres_sw_all+ceres_lw_all), start=start, end=end)
merra_crf_net.anom<-anom(merra_crf_net, 12)
merra_crf_sw.anom<-anom(window(merra_sw_clr-ceres_sw_all, start=start, end=end),12)
merra_crf_lw.anom<-anom(window(merra_lw_clr-ceres_lw_all, start=start, end=end),12)


#Determine long-wave clear-sky from AIRS_CAM
AIRS_ClrOLR<-read.table("AIRS_clrOLR.txt")
clear_sky.airs<-ts(AIRS_ClrOLR[[2]], start=c(2002,9), freq=12)
ceres_airs_lw.anom<-anom(anom(clear_sky.airs)-anom(ceres_lw_all))

aquaMatrix<-array(0, dim=c(length(tempFiles)*7, 3))

for (i in 1:length(tempFiles))
{

	temp.ts<-ts(read.table(tempFiles[i])[[2]], start=tempStartDates[[i]], freq=12)
	temp.anom<-anom(window(temp.ts, start=start, end=endDates[[i]]))

	#First do them for CRF (unadjusted regressions)
	TERRA.monthly<-getRegressionRow(window(terra_crf_net.anom, end=endDates[[i]]), window(terra_crf_sw.anom, end=endDates[[i]]), window(terra_crf_lw.anom, end=endDates[[i]]), temp.anom, 1)
	AQUA.monthly<-getRegressionRow(window(aqua_crf_net.anom, end=endDates[[i]]), window(aqua_crf_sw.anom, end=endDates[[i]]), window(aqua_crf_lw.anom, end=endDates[[i]]), temp.anom, 1)
	CERES.monthly<-getRegressionRow(window(ceres_crf_net.anom, end=endDates[[i]]), window(ceres_crf_sw.anom, end=endDates[[i]]), window(ceres_crf_lw.anom, end=endDates[[i]]), temp.anom, 1)
	ERA.monthly<-getRegressionRow(window(era_crf_net.anom, end=endDates[[i]]), window(era_crf_sw.anom, end=endDates[[i]]), window(era_crf_lw.anom, end=endDates[[i]]), temp.anom, 1)
	EBAF.monthly<-getRegressionRow(window(ebaf_crf_net.anom, end=endDates[[i]]), window(ebaf_crf_sw.anom, end=endDates[[i]]), window(ebaf_crf_lw.anom, end=endDates[[i]]), temp.anom, 1)
	AIRS.monthly<-getRegressionRowLw(window(ceres_airs_lw.anom, end=endDates[[i]]), temp.anom, 1)
	MERRA.monthly<-getRegressionRow(window(merra_crf_net.anom, end=endDates[[i]]), window(merra_crf_sw.anom, end=endDates[[i]]), window(merra_crf_lw.anom, end=endDates[[i]]), temp.anom, 1)
	
	tblMatrix1<-rbind(TERRA.monthly, AQUA.monthly, CERES.monthly, ERA.monthly, EBAF.monthly, MERRA.monthly, AIRS.monthly)

	#Perform kernel adjustments to go from CRF values to R_cld values
	terra_rcloud_net.anom<-anom(window(doKernelAdjust(terra_crf_net.anom, TRUE, TRUE, TRUE), end=endDates[[i]])) 
	terra_rcloud_sw.anom<-anom(window(doKernelAdjust(terra_crf_sw.anom, TRUE, FALSE, TRUE), end=endDates[[i]])) 
	terra_rcloud_lw.anom<-anom(window(doKernelAdjust(terra_crf_lw.anom, TRUE, TRUE, FALSE), end=endDates[[i]])) 
	aqua_rcloud_net.anom<-anom(window(doKernelAdjust(aqua_crf_net.anom, TRUE, TRUE, TRUE), end=endDates[[i]])) 
	aqua_rcloud_sw.anom<-anom(window(doKernelAdjust(aqua_crf_sw.anom, TRUE, FALSE, TRUE), end=endDates[[i]])) 
	aqua_rcloud_lw.anom<-anom(window(doKernelAdjust(aqua_crf_lw.anom, TRUE, TRUE, FALSE), end=endDates[[i]])) 	
	ceres_rcloud_net.anom<-anom(window(doKernelAdjust(ceres_crf_net.anom, TRUE, TRUE, TRUE), end=endDates[[i]])) 
	ceres_rcloud_sw.anom<-anom(window(doKernelAdjust(ceres_crf_sw.anom, TRUE, FALSE, TRUE), end=endDates[[i]])) 
	ceres_rcloud_lw.anom<-anom(window(doKernelAdjust(ceres_crf_lw.anom, TRUE, TRUE, FALSE), end=endDates[[i]])) 
	ebaf_rcloud_net.anom<-anom(window(doKernelAdjust(ebaf_crf_net.anom, TRUE, TRUE, TRUE), end=endDates[[i]])) 
	ebaf_rcloud_sw.anom<-anom(window(doKernelAdjust(ebaf_crf_sw.anom, TRUE, FALSE, TRUE), end=endDates[[i]])) 
	ebaf_rcloud_lw.anom<-anom(window(doKernelAdjust(ebaf_crf_lw.anom, TRUE, TRUE, FALSE), end=endDates[[i]])) 
	era_rcloud_net.anom<-anom(window(doKernelAdjust(era_crf_net.anom, TRUE, TRUE, TRUE), end=endDates[[i]])) 
	era_rcloud_sw.anom<-anom(window(doKernelAdjust(era_crf_sw.anom, TRUE, FALSE, TRUE), end=endDates[[i]])) 
	era_rcloud_lw.anom<-anom(window(doKernelAdjust(era_crf_lw.anom, TRUE, TRUE, FALSE), end=endDates[[i]])) 
	merra_rcloud_net.anom<-anom(window(doKernelAdjust(merra_crf_net.anom, TRUE, TRUE, TRUE), end=endDates[[i]])) 
	merra_rcloud_sw.anom<-anom(window(doKernelAdjust(merra_crf_sw.anom, TRUE, FALSE, TRUE), end=endDates[[i]])) 
	merra_rcloud_lw.anom<-anom(window(doKernelAdjust(merra_crf_lw.anom, TRUE, TRUE, FALSE), end=endDates[[i]])) 
	airs_rcloud_lw.anom<-anom(window(doKernelAdjust(ceres_airs_lw.anom, TRUE, TRUE, FALSE), end=endDates[[i]]))

	TERRA.monthly<-getRegressionRow(terra_rcloud_net.anom, terra_rcloud_sw.anom, terra_rcloud_lw.anom, temp.anom, 1)
	AQUA.monthly<-getRegressionRow(aqua_rcloud_net.anom, aqua_rcloud_sw.anom, aqua_rcloud_lw.anom, temp.anom, 1)
	CERES.monthly<-getRegressionRow(ceres_rcloud_net.anom, ceres_rcloud_sw.anom, ceres_rcloud_lw.anom, temp.anom, 1)
	ERA.monthly<-getRegressionRow(era_rcloud_net.anom, era_rcloud_sw.anom, era_rcloud_lw.anom, temp.anom, 1)
	EBAF.monthly<-getRegressionRow(ebaf_rcloud_net.anom, ebaf_rcloud_sw.anom, ebaf_rcloud_lw.anom, temp.anom, 1)
	MERRA.monthly<-getRegressionRow(merra_rcloud_net.anom, merra_rcloud_sw.anom, merra_rcloud_lw.anom, temp.anom, 1)
	AIRS.monthly<-getRegressionRowLw(airs_rcloud_lw.anom, temp.anom, 1)

	tblMatrix2<-rbind(TERRA.monthly, AQUA.monthly, CERES.monthly, ERA.monthly, EBAF.monthly, MERRA.monthly, AIRS.monthly) 
	
	#here we merge the two regression tables (R_cld vs. Temp and CRF vs. Temp) with (CRF vs. Temp in parenthesis)
	tblMatrix<-array(0, dim=c(nrow(tblMatrix1), ncol(tblMatrix1)))
	for (row in 1:nrow(tblMatrix1))
	{
		for (col in 1:ncol(tblMatrix1))
		{
			tblMatrix[row,col]<-paste(tblMatrix2[row,col], "  (", tblMatrix1[row,col], ")", sep='')
		}
	}

	aquaMatrix[((i-1)*7+1):(i*7), 1:3]<-tblMatrix
}

##output results	
df<-data.frame(aquaMatrix)
for (i in 1:length(tempFiles))
{
	rownames(df)[((i-1)*7+1)]<-paste("TERRA and", tempFiles[i])
	rownames(df)[((i-1)*7+2)]<-paste("AQUA and", tempFiles[i])
	rownames(df)[((i-1)*7+3)]<-paste("CERES and", tempFiles[i])
	rownames(df)[((i-1)*7+4)]<-paste("ERA and", tempFiles[i])
	rownames(df)[((i-1)*7+5)]<-paste("EBAF and", tempFiles[i])
	rownames(df)[((i-1)*7+6)]<-paste("MERRA and", tempFiles[i])
	rownames(df)[((i-1)*7+7)]<-paste("AIRS and", tempFiles[i])
}
write.table(df, file="CloudFeedback_AQUA_temps.txt",sep="\t")

}


######Actual call to run

tempFiles<-c("GISTemp.txt", "NCDC_temps.txt", "HadCRUT3_temps.txt", "hadCRUT4_temps.txt", "era_t2m.txt", "merra_t2m.txt", "ncep_t2m.txt")
tempStarts<-list(1880, 1880, 1850, 1850, 2000, 1979, 1948)
endDates<-list(c(2011,6), c(2011,6), c(2011,6), c(2010,12),c(2011,6), c(2011,6), c(2011,6))

createAnalysisTables(tempFiles, tempStarts, endDates)




