#Troy Masters 2012

#set working directory here
setwd("C:/Climate/Masters_2012")

library(ncdf) #for reading netCDF files

#end of period (do full length of CERES available obs)
end<-c(2011,6)

#########################
#HELPER FUNCTIONS
############################
#Calculate the anomaly based on the yearly frequency
anom<-function(x, freq=12)
{
	totalSum<-numeric(freq)
	numObs<-numeric(freq)
	for (mon in 1:length(x))
	{
		totalSum[mon %% freq + 1]<-totalSum[mon %% freq + 1] + x[mon]
		numObs[mon %% freq + 1]<-numObs[mon %% freq + 1] + 1
	}
	anom<-numeric(length(x))
	for (mon in 1:length(x))
	{
		anom[mon]<-x[mon]-(totalSum[mon %% freq + 1]/numObs[mon %% freq + 1])	
	}	
	anom<-ts(anom,start=tsp(x)[1],freq=freq) 
}

#get nicely formatted row of data based on regression of CRF against temperatures
##################################
getRegressionRow<-function(fluxAll, fluxSw, fluxLw, temps, monthsAgg)
{
	freq<-12/monthsAgg

        fluxAll.agg<-anom(aggregate(fluxAll, nfrequency=freq)/monthsAgg, freq)
	fluxSw.agg<-anom(aggregate(fluxSw, nfrequency=freq)/monthsAgg, freq)
	fluxLw.agg<-anom(aggregate(fluxLw, nfrequency=freq)/monthsAgg, freq)
	temps.agg<-anom(aggregate(temps, nfrequency=freq)/monthsAgg, freq)	

	regNet.lm<-lm(fluxAll.agg ~ temps.agg, data=ts.union(fluxAll.agg, temps.agg))
	regSw.lm<-lm(fluxSw.agg ~ temps.agg, data=ts.union(fluxSw.agg, temps.agg))
	regLw.lm<-lm(fluxLw.agg ~ temps.agg, data=ts.union(fluxLw.agg, temps.agg))
	
	#output our R^2 values to console
	print(summary(regNet.lm)$r.squared)
	print(summary(regSw.lm)$r.squared)
	print(summary(regLw.lm)$r.squared)

	y1<-paste(format(round(coef(regNet.lm)[2],2), nsmall=2), "", format(round(coef(regNet.lm)[2]-confint(regNet.lm)[2,1], 2), nsmall=2)) 
	y2<-paste(format(round(coef(regSw.lm)[2],2), nsmall=2), "", format(round(coef(regSw.lm)[2]-confint(regSw.lm)[2,1], 2), nsmall=2))
	y3<-paste(format(round(coef(regLw.lm)[2],2), nsmall=2), "", format(round(coef(regLw.lm)[2]-confint(regLw.lm)[2,1], 2), nsmall=2))
	regRow<-c(y1, y2, y3)
}

getRegressionRowLw<-function(fluxLw, temps, monthsAgg)
{
        fluxLw.agg<-anom(aggregate(fluxLw, nfrequency=12/monthsAgg)/monthsAgg, 12/monthsAgg)
	temps.agg<-anom(aggregate(temps, nfrequency=12/monthsAgg)/monthsAgg, 12/monthsAgg)	
	regLw.lm<-lm(fluxLw.agg ~ temps.agg, data=ts.union(fluxLw.agg, temps.agg))

	#output our R^2 values to console
	print(summary(regLw.lm)$r.squared)

	y3<-paste(format(round(coef(regLw.lm)[2],2), nsmall=2), "", format(round(coef(regLw.lm)[2]-confint(regLw.lm)[2,1], 2), nsmall=2))
	regRow<-c(NA, NA, y3)
}

#doKernelAdjust - perform kernel adjustments
#crf.anom - CRF time series of anomalies
#useAirs - TRUE use AIRS (only starts 9-2002) for LW, FALSE user ERA-Interim LW adjusts
#lwAdjust - TRUE perform adjustments to OLR
#swAdjust - TRUE perform adjustments to SW 
#returns - CRF adjusted to become R_Cloud
doKernelAdjust<-function(crf.anom, useAirs, lwAdjust, swAdjust)
{
	rcloud.anom<-crf.anom
	era_fluxes<-read.table("ERA_GFDL_GlobalFluxContributions_2000_1-2011_11.txt")
	airs_fluxes<-read.table("AIRS_GFDL_GlobalFluxContributions_2002_9-2011_10.txt")
	if (!useAirs) #need full period adjustments
	{
		surfaceTempFluxAdj<-ts(era_fluxes[[2]]-era_fluxes[[7]], start=c(2000,1), freq=12)
		atmoTempFluxAdj<-ts(era_fluxes[[3]]-era_fluxes[[8]], start=c(2000,1), freq=12)
		lwWvFluxAdj<-ts(era_fluxes[[4]]-era_fluxes[[9]], start=c(2000,1), freq=12)
		swWvFluxAdj<-ts(era_fluxes[[10]]-era_fluxes[[5]], start=c(2000,1), freq=12)
	}
	if (useAirs) #only need to adjust over AIRS period
	{
		surfaceTempFluxAdj<-ts(airs_fluxes[[1]]-airs_fluxes[[5]], start=c(2002,9), freq=12)
		atmoTempFluxAdj<-ts(airs_fluxes[[2]]-airs_fluxes[[6]], start=c(2002,9), freq=12)
		lwWvFluxAdj<-ts(airs_fluxes[[3]]-airs_fluxes[[7]], start=c(2002,9), freq=12)
		swWvFluxAdj<-ts(airs_fluxes[[8]]-airs_fluxes[[4]], start=c(2002,9), freq=12)
	}	

	albedoFluxAdj<-ts(era_fluxes[[1]]-era_fluxes[[6]], start=c(2000,1), freq=12)
	CO2Adjust<--1*ts((1:140)/140*0.25 * 0.16, start=c(2000,3), freq=12)
	if (lwAdjust)
	{
		rcloud.anom<-rcloud.anom + surfaceTempFluxAdj +atmoTempFluxAdj + CO2Adjust + lwWvFluxAdj
	}
	if (swAdjust)
	{
		rcloud.anom<-rcloud.anom + swWvFluxAdj - albedoFluxAdj
	}	
	rcloud.anom	
}

###########################################
#1. PERFORM ANALYSIS OVER TERRA PERIOD
###########################################
start<-c(2000,3)

terraStart<-c(2000,3)

#Read TERRA clear and all-sky
terra.nc<-open.ncdf("CERES_SSF1deg-Month-lite_Terra_Ed2.6_Subset_200003-201106.nc")
terra_net_all<-ts(get.var.ncdf(terra.nc, "gtoa_net_all_mon"), start=terraStart, freq=12)
terra_sw_all<-ts(get.var.ncdf(terra.nc, "gtoa_sw_all_mon"), start=terraStart, freq=12)
terra_sw_clr<-ts(get.var.ncdf(terra.nc, "gtoa_sw_clr_mon"), start=terraStart, freq=12)
terra_lw_all<-ts(get.var.ncdf(terra.nc, "gtoa_lw_all_mon"), start=terraStart, freq=12)
terra_lw_clr<-ts(get.var.ncdf(terra.nc, "gtoa_lw_clr_mon"), start=terraStart, freq=12)
close.ncdf(terra.nc)

#Read EBAF clear-sky
ebaf.nc<-open.ncdf("CERES_EBAF-TOA_Ed2.6r_Subset_200003-201106.nc")
ebaf_net_clr<-ts(get.var.ncdf(ebaf.nc, "gtoa_net_clr_mon"), start=terraStart, freq=12)
ebaf_sw_clr<-ts(get.var.ncdf(ebaf.nc, "gtoa_sw_clr_mon"), start=terraStart, freq=12)
ebaf_lw_clr<-ts(get.var.ncdf(ebaf.nc, "gtoa_lw_clr_mon"), start=terraStart, freq=12)
close.ncdf(ebaf.nc)

#get ERA clear-sky data
eraData<-read.table("era_flux_data.txt", header=TRUE)
era_sw_inc<-ts(eraData["sw_inc"], freq=12, start=2000)
era_sw_clr<-ts(eraData["sw_out_clr"], freq=12, start=2000)
era_lw_clr<-ts(eraData["lw_out_clr"], freq=12, start=2000)

####Calculate CRF values
#First for TERRA
terra_crf_net<-window((terra_sw_clr+terra_lw_clr)-(terra_sw_all+terra_lw_all), start=start, end=end)
terra_crf_net.anom<-anom(terra_crf_net)
terra_crf_sw.anom<-anom(window(terra_sw_clr-terra_sw_all, start=start, end=end),12)
terra_crf_lw.anom<-anom(window(terra_lw_clr-terra_lw_all, start=start, end=end),12)

#Then for ERA-Interim
####combine just sw and lw to avoid incident solar inconsistency
era_crf_net<-window((era_sw_clr+era_lw_clr)-(terra_sw_all+terra_lw_all), start=start, end=end)
era_crf_net.anom<-anom(era_crf_net, 12)
era_crf_sw.anom<-anom(window(era_sw_clr-terra_sw_all, start=start, end=end),12)
era_crf_lw.anom<-anom(window(era_lw_clr-terra_lw_all, start=start, end=end),12)

#Then for EBAF
ebaf_crf_net<-window((ebaf_sw_clr+ebaf_lw_clr)-(terra_sw_all+terra_lw_all), start=start, end=end)
ebaf_crf_net.anom<-anom(ebaf_crf_net, 12)
ebaf_crf_sw.anom<-anom(window(ebaf_sw_clr-terra_sw_all, start=start, end=end),12)
ebaf_crf_lw.anom<-anom(window(ebaf_lw_clr-terra_lw_all, start=start, end=end),12)

##Get our temperature records
GISS.ts<-ts(read.table("GISTemp.txt")[[2]], start=1880, freq=12)
GISS.ts<-anom(window(GISS.ts, start=start, end=end))

ERA_temp.ts<-ts(read.table("era_t2m.txt")[[2]], start=2000, freq=12)
ERA_temp.ts<-anom(window(ERA_temp.ts, start=start, end=end))

ERA_skin.ts<-ts(read.table("era_skt.txt")[[2]], start=2000, freq=12)
ERA_skin.ts<-anom(window(ERA_skin.ts, start=start, end=end))

################################
# FIGURE 1: Terra all-sky, then CERES & ERA clear-sky, then CERES & ERA CRF
#all-sky
ts.plot(anom(terra_net_all), ylab=expression(Delta~R[all]~'('~Wm^{-2}~')'))
#clear-skies
ts.plot(anom(window(-1*(era_sw_clr+era_lw_clr),start=start,end=end)), anom(window(-1*(ebaf_sw_clr+ebaf_lw_clr),start=start,end=end)), 
	anom(window(-1*(terra_sw_clr+terra_lw_clr),start=start,end=end)), col=c("red", "green", "black"), ylab=expression(Delta~R[clr]~'('~Wm^{-2}~')'))
legend("top", legend=c("CERES Terra", "ECMWF", "EBAF"), col=c("black", "red", "green"), lty=1)

#CRF
ts.plot(era_crf_net.anom, ebaf_crf_net.anom, terra_crf_net.anom, col=c("red", "green", "black"),
	ylab=expression(~Delta~CRF~' ('~Wm^{-2}~')'), lwd=1)
legend("bottom", legend=c("CERES Only", "CERES - ECMWF", "CERES - EBAF"), col=c("black", "red", "green"), lty=1)

################################
# TABLE 1 Estimates

#First do them for CRF (unadjusted regressions)
#Against GISS
GISS_CERES.monthly<-getRegressionRow(terra_crf_net.anom, terra_crf_sw.anom, terra_crf_lw.anom, GISS.ts, 1)
GISS_CERES.seasonal<-getRegressionRow(terra_crf_net.anom, terra_crf_sw.anom, terra_crf_lw.anom, GISS.ts, 3)
GISS_ERA.monthly<-getRegressionRow(era_crf_net.anom, era_crf_sw.anom, era_crf_lw.anom, GISS.ts, 1)
GISS_ERA.seasonal<-getRegressionRow(era_crf_net.anom, era_crf_sw.anom, era_crf_lw.anom, GISS.ts, 3)
GISS_EBAF.monthly<-getRegressionRow(ebaf_crf_net.anom, ebaf_crf_sw.anom, ebaf_crf_lw.anom, GISS.ts, 1)
GISS_EBAF.seasonal<-getRegressionRow(ebaf_crf_net.anom, ebaf_crf_sw.anom, ebaf_crf_lw.anom, GISS.ts, 3)

#Against ERA-t2m
ERAT_CERES.monthly<-getRegressionRow(terra_crf_net.anom, terra_crf_sw.anom, terra_crf_lw.anom, ERA_temp.ts, 1)
ERAT_CERES.seasonal<-getRegressionRow(terra_crf_net.anom, terra_crf_sw.anom, terra_crf_lw.anom, ERA_temp.ts, 3)
ERAT_ERA.monthly<-getRegressionRow(era_crf_net.anom, era_crf_sw.anom, era_crf_lw.anom, ERA_temp.ts, 1)
ERAT_ERA.seasonal<-getRegressionRow(era_crf_net.anom, era_crf_sw.anom, era_crf_lw.anom, ERA_temp.ts, 3)
ERAT_EBAF.monthly<-getRegressionRow(ebaf_crf_net.anom, ebaf_crf_sw.anom, ebaf_crf_lw.anom, ERA_temp.ts, 1)
ERAT_EBAF.seasonal<-getRegressionRow(ebaf_crf_net.anom, ebaf_crf_sw.anom, ebaf_crf_lw.anom, ERA_temp.ts, 3)

#Against ERA-skt
ERASKT_CERES.monthly<-getRegressionRow(terra_crf_net.anom, terra_crf_sw.anom, terra_crf_lw.anom, ERA_skin.ts, 1)
ERASKT_CERES.seasonal<-getRegressionRow(terra_crf_net.anom, terra_crf_sw.anom, terra_crf_lw.anom, ERA_skin.ts, 3)
ERASKT_ERA.monthly<-getRegressionRow(era_crf_net.anom, era_crf_sw.anom, era_crf_lw.anom, ERA_skin.ts, 1)
ERASKT_ERA.seasonal<-getRegressionRow(era_crf_net.anom, era_crf_sw.anom, era_crf_lw.anom, ERA_skin.ts, 3)
ERASKT_EBAF.monthly<-getRegressionRow(ebaf_crf_net.anom, ebaf_crf_sw.anom, ebaf_crf_lw.anom, ERA_skin.ts, 1)
ERASKT_EBAF.seasonal<-getRegressionRow(ebaf_crf_net.anom, ebaf_crf_sw.anom, ebaf_crf_lw.anom, ERA_skin.ts, 3)

tblMatrix1<-rbind(c(GISS_CERES.monthly, GISS_CERES.seasonal), c(ERAT_CERES.monthly,ERAT_CERES.seasonal),
		 c(ERASKT_CERES.monthly, ERASKT_CERES.seasonal),
		 c(GISS_ERA.monthly, GISS_ERA.seasonal),c(ERAT_ERA.monthly, ERAT_ERA.seasonal),
		 c(ERASKT_ERA.monthly, ERASKT_ERA.seasonal),
		 c(GISS_EBAF.monthly, GISS_EBAF.seasonal), c(ERAT_EBAF.monthly, ERAT_EBAF.seasonal),
		 c(ERASKT_EBAF.monthly, ERASKT_EBAF.seasonal))

#Perform kernel adjustments to go from CRF values to R_cld values
terra_rcloud_net.anom<-anom(doKernelAdjust(terra_crf_net.anom, FALSE, TRUE, TRUE)) 
terra_rcloud_sw.anom<-anom(doKernelAdjust(terra_crf_sw.anom, FALSE, FALSE, TRUE)) 
terra_rcloud_lw.anom<-anom(doKernelAdjust(terra_crf_lw.anom, FALSE, TRUE, FALSE)) 
ebaf_rcloud_net.anom<-anom(doKernelAdjust(ebaf_crf_net.anom, FALSE, TRUE, TRUE)) 
ebaf_rcloud_sw.anom<-anom(doKernelAdjust(ebaf_crf_sw.anom, FALSE, FALSE, TRUE)) 
ebaf_rcloud_lw.anom<-anom(doKernelAdjust(ebaf_crf_lw.anom, FALSE, TRUE, FALSE)) 
era_rcloud_net.anom<-anom(doKernelAdjust(era_crf_net.anom, FALSE, TRUE, TRUE)) 
era_rcloud_sw.anom<-anom(doKernelAdjust(era_crf_sw.anom, FALSE, FALSE, TRUE)) 
era_rcloud_lw.anom<-anom(doKernelAdjust(era_crf_lw.anom, FALSE, TRUE, FALSE)) 

#Now do the regressions against R_cloud
#First GISS
GISS_CERES.monthly<-getRegressionRow(terra_rcloud_net.anom, terra_rcloud_sw.anom, terra_rcloud_lw.anom, GISS.ts, 1)
GISS_CERES.seasonal<-getRegressionRow(terra_rcloud_net.anom, terra_rcloud_sw.anom, terra_rcloud_lw.anom, GISS.ts, 3)
GISS_ERA.monthly<-getRegressionRow(era_rcloud_net.anom, era_rcloud_sw.anom, era_rcloud_lw.anom, GISS.ts, 1)
GISS_ERA.seasonal<-getRegressionRow(era_rcloud_net.anom, era_rcloud_sw.anom, era_rcloud_lw.anom, GISS.ts, 3)
GISS_EBAF.monthly<-getRegressionRow(ebaf_rcloud_net.anom, ebaf_rcloud_sw.anom, ebaf_rcloud_lw.anom, GISS.ts, 1)
GISS_EBAF.seasonal<-getRegressionRow(ebaf_rcloud_net.anom, ebaf_rcloud_sw.anom, ebaf_rcloud_lw.anom, GISS.ts, 3)

#Then ERA t2m
ERAT_CERES.monthly<-getRegressionRow(terra_rcloud_net.anom, terra_rcloud_sw.anom, terra_rcloud_lw.anom, ERA_temp.ts, 1)
ERAT_CERES.seasonal<-getRegressionRow(terra_rcloud_net.anom, terra_rcloud_sw.anom, terra_rcloud_lw.anom, ERA_temp.ts, 3)
ERAT_ERA.monthly<-getRegressionRow(era_rcloud_net.anom, era_rcloud_sw.anom, era_rcloud_lw.anom, ERA_temp.ts, 1)
ERAT_ERA.seasonal<-getRegressionRow(era_rcloud_net.anom, era_rcloud_sw.anom, era_rcloud_lw.anom, ERA_temp.ts, 3)
ERAT_EBAF.monthly<-getRegressionRow(ebaf_rcloud_net.anom, ebaf_rcloud_sw.anom, ebaf_rcloud_lw.anom, ERA_temp.ts, 1)
ERAT_EBAF.seasonal<-getRegressionRow(ebaf_rcloud_net.anom, ebaf_rcloud_sw.anom, ebaf_rcloud_lw.anom, ERA_temp.ts, 3)

#Then ERA skt
ERASKT_CERES.monthly<-getRegressionRow(terra_rcloud_net.anom, terra_rcloud_sw.anom, terra_rcloud_lw.anom, ERA_skin.ts, 1)
ERASKT_CERES.seasonal<-getRegressionRow(terra_rcloud_net.anom, terra_rcloud_sw.anom, terra_rcloud_lw.anom, ERA_skin.ts, 3)
ERASKT_ERA.monthly<-getRegressionRow(era_rcloud_net.anom, era_rcloud_sw.anom, era_rcloud_lw.anom, ERA_skin.ts, 1)
ERASKT_ERA.seasonal<-getRegressionRow(era_rcloud_net.anom, era_rcloud_sw.anom, era_rcloud_lw.anom, ERA_skin.ts, 3)
ERASKT_EBAF.monthly<-getRegressionRow(ebaf_rcloud_net.anom, ebaf_rcloud_sw.anom, ebaf_rcloud_lw.anom, ERA_skin.ts, 1)
ERASKT_EBAF.seasonal<-getRegressionRow(ebaf_rcloud_net.anom, ebaf_rcloud_sw.anom, ebaf_rcloud_lw.anom, ERA_skin.ts, 3)



tblMatrix2<-rbind(c(GISS_CERES.monthly, GISS_CERES.seasonal), c(ERAT_CERES.monthly,ERAT_CERES.seasonal),
		 c(ERASKT_CERES.monthly, ERASKT_CERES.seasonal),
		 c(GISS_ERA.monthly, GISS_ERA.seasonal),c(ERAT_ERA.monthly, ERAT_ERA.seasonal),
		 c(ERASKT_ERA.monthly, ERASKT_ERA.seasonal),
		 c(GISS_EBAF.monthly, GISS_EBAF.seasonal), c(ERAT_EBAF.monthly, ERAT_EBAF.seasonal),
		 c(ERASKT_EBAF.monthly, ERASKT_EBAF.seasonal))

#here we merge the two regression tables (R_cld vs. Temp and CRF vs. Temp) with (CRF vs. Temp in parenthesis)
tblMatrix<-array(0, dim=c(nrow(tblMatrix1), ncol(tblMatrix1)))
for (row in 1:nrow(tblMatrix1))
{
	for (col in 1:ncol(tblMatrix1))
	{
		tblMatrix[row,col]<-paste(tblMatrix2[row,col], "  (", tblMatrix1[row,col], ")", sep='')
	}
}
	
##output results	
df<-data.frame(tblMatrix)
rownames(df)[1]<-"GISTemp and CERES_CRF"
rownames(df)[2]<-"ERA-T and CERES_CRF"
rownames(df)[3]<-"ERA-SKT and CERES_CRF"
rownames(df)[4]<-"GISTemp and ERA_CRF"
rownames(df)[5]<-"ERA-T and ERA_CRF"
rownames(df)[6]<-"ERA-SKT and ERA_CRF"
rownames(df)[7]<-"GISTemp and EBAF_CRF"
rownames(df)[8]<-"ERA-T and EBAF_CRF"
rownames(df)[9]<-"ERA-SKT and EBAF_CRF"
write.table(df, file="CloudFeedback_TERRA2.txt",sep="\t")

##Sensitivity 
ceresRegs<-numeric(88)
eraRegs<-numeric(88)
ebafRegs<-numeric(88)
temps<-GISS.ts
for (i in 1:88)
{
	temp.ts<-anom(ts(temps[i:length(temps)], freq=12))
	ceres_crf.ts<-anom(ts(terra_crf_net.anom[i:length(temps)], freq=12))
	reg1.lm<-lm(ceres_crf.ts ~ temp.ts)
	ceresRegs[i]<-coef(reg1.lm)[2]
	era_crf.ts<-anom(ts(era_crf_net.anom[i:length(temps)], freq=12))
	reg2.lm<-lm(era_crf.ts ~ temp.ts)
	eraRegs[i]<-coef(reg2.lm)[2]	
	ebaf_crf.ts<-anom(ts(ebaf_crf_net.anom[i:length(temps)], freq=12))
	reg3.lm<-lm(ebaf_crf.ts ~ temp.ts)
	ebafRegs[i]<-coef(reg3.lm)[2]		
}
ceresRegs.ts<-ts(ceresRegs, start=start, freq=12)
eraRegs.ts<-ts(eraRegs, start=start, freq=12)
ebafRegs.ts<-ts(ebafRegs, start=start, freq=12)
ts.plot(ceresRegs.ts, eraRegs.ts, ebafRegs.ts, col=c("black", "red", "green"), xlab="Start Date", ylab=expression(frac(dCRF,dT)~' ('~Wm^{-2}~K^{-1}~')'))
abline(h=0, lty=2)
legend("bottom", legend=c("CERES Terra", "CERES-ECMWF", "CERES-EBAF"), col=c("black", "red", "green"), lty=1) 


###########################################
#2. PERFORM ANALYSIS OVER AQUA PERIOD
###########################################
start<-c(2002,9)
aquaStart<-c(2002,7)

aqua.nc<-open.ncdf("CERES_SSF1deg-Month-lite_Aqua_Ed2.6_Subset_200207-201106.nc") #open our nc file
aqua_sw_all<-ts(get.var.ncdf(aqua.nc, "gtoa_sw_all_mon"), start=aquaStart, freq=12)
aqua_sw_clr<-ts(get.var.ncdf(aqua.nc, "gtoa_sw_clr_mon"), start=aquaStart, freq=12)
aqua_lw_all<-ts(get.var.ncdf(aqua.nc, "gtoa_lw_all_mon"), start=aquaStart, freq=12)
aqua_lw_clr<-ts(get.var.ncdf(aqua.nc, "gtoa_lw_clr_mon"), start=aquaStart, freq=12)
close.ncdf(aqua.nc)

#here we average Terra and Aqua satellite observations
ceres_sw_all<-(aqua_sw_all+window(terra_sw_all, start=aquaStart))/2
ceres_sw_clr<-(aqua_sw_clr+window(terra_sw_clr, start=aquaStart))/2
ceres_lw_all<-(aqua_lw_all+window(terra_lw_all, start=aquaStart))/2
ceres_lw_clr<-(aqua_lw_clr+window(terra_lw_clr, start=aquaStart))/2


####Calculate CRF values
ceres_crf_net<-window((ceres_sw_clr+ceres_lw_clr)-(ceres_sw_all+ceres_lw_all), start=start, end=end)
ceres_crf_net.anom<-anom(ceres_crf_net)
ceres_crf_sw.anom<-anom(window(ceres_sw_clr-ceres_sw_all, start=start, end=end),12)
ceres_crf_lw.anom<-anom(window(ceres_lw_clr-ceres_lw_all, start=start, end=end),12)

##re-calculate other anomalies
era_crf_net<-window((era_sw_clr+era_lw_clr)-(ceres_sw_all+ceres_lw_all), start=start, end=end)
era_crf_net.anom<-anom(era_crf_net, 12)
era_crf_sw.anom<-anom(window(era_sw_clr-ceres_sw_all, start=start, end=end),12)
era_crf_lw.anom<-anom(window(era_lw_clr-ceres_lw_all, start=start, end=end),12)

#Then for EBAF
ebaf_crf_net<-window((ebaf_sw_clr+ebaf_lw_clr)-(ceres_sw_all+ceres_lw_all), start=start, end=end)
ebaf_crf_net.anom<-anom(ebaf_crf_net, 12)
ebaf_crf_sw.anom<-anom(window(ebaf_sw_clr-ceres_sw_all, start=start, end=end),12)
ebaf_crf_lw.anom<-anom(window(ebaf_lw_clr-ceres_lw_all, start=start, end=end),12)

#temperature series
GISS.ts<-anom(window(GISS.ts, start=start, end=end))
ERA_temp.ts<-anom(window(ERA_temp.ts, start=start, end=end))
ERA_skin.ts<-anom(window(ERA_skin.ts, start=start, end=end))

#Determine long-wave clear-sky from AIRS_CAM
AIRS_ClrOLR<-read.table("AIRS_clrOLR.txt")
clear_sky.airs<-ts(AIRS_ClrOLR[[2]], start=c(2002,9), freq=12)
ceres_airs_lw.anom<-anom(anom(clear_sky.airs)-anom(ceres_lw_all))

################################
# FIGURE 2: CERES & ERA clear-sky
#short-wave
ts.plot(anom(window(-1*(era_sw_clr),start=start,end=end)), anom(window(-1*(terra_sw_clr+aqua_sw_clr)/2,start=start,end=end)), 
	anom(window(-1*(ebaf_sw_clr),start=start,end=end)), col=c("red", "black", "green"), ylab=expression(Delta~R[SW~','~CLR]~'('~Wm^{-2}~')'))
legend("top", legend=c("CERES", "ECMWF", "EBAF"), col=c("black", "red", "green"), lty=1)

#long-wave
ts.plot(anom(window(-1*(era_lw_clr),start=start,end=end)), anom(window(-1*(terra_lw_clr+aqua_lw_clr)/2,start=start,end=end)), 
	anom(window(-1*clear_sky.airs, start=start,end=end)), anom(window(-1*(ebaf_lw_clr),start=start,end=end)),
	col=c("red", "black", "blue", "green"), ylab=expression(Delta~R[LW~','~CLR]~'('~Wm^{-2}~')'))
legend("bottom", legend=c("CERES", "ECMWF", "AIRS", "EBAF"), col=c("black", "red", "blue", "green"), lty=1)

#surface temperatures
ts.plot(GISS.ts, ERA_temp.ts, ERA_skin.ts, col=c("black", "red", "blue"), ylab=expression(Delta~T[surface]))
legend("bottomleft", legend=c("GISS", "ECMWF 2-Meter", "ECMWF Skin"), col=c("black", "red", "blue"), lty=1)


################################
# TABLE 2 Estimates

#First do them for CRF (unadjusted regressions)
#Against GISS
GISS_CERES.monthly<-getRegressionRow(ceres_crf_net.anom, ceres_crf_sw.anom, ceres_crf_lw.anom, GISS.ts, 1)
GISS_CERES.seasonal<-getRegressionRow(ceres_crf_net.anom, ceres_crf_sw.anom, ceres_crf_lw.anom, GISS.ts, 3)
GISS_ERA.monthly<-getRegressionRow(era_crf_net.anom, era_crf_sw.anom, era_crf_lw.anom, GISS.ts, 1)
GISS_ERA.seasonal<-getRegressionRow(era_crf_net.anom, era_crf_sw.anom, era_crf_lw.anom, GISS.ts, 3)
GISS_EBAF.monthly<-getRegressionRow(ebaf_crf_net.anom, ebaf_crf_sw.anom, ebaf_crf_lw.anom, GISS.ts, 1)
GISS_EBAF.seasonal<-getRegressionRow(ebaf_crf_net.anom, ebaf_crf_sw.anom, ebaf_crf_lw.anom, GISS.ts, 3)
GISS_AIRS.monthly<-getRegressionRowLw(ceres_airs_lw.anom, GISS.ts, 1)
GISS_AIRS.seasonal<-getRegressionRowLw(ceres_airs_lw.anom, GISS.ts, 3)

#Against ERA-t2m
ERAT_CERES.monthly<-getRegressionRow(ceres_crf_net.anom, ceres_crf_sw.anom, ceres_crf_lw.anom, ERA_temp.ts, 1)
ERAT_CERES.seasonal<-getRegressionRow(ceres_crf_net.anom, ceres_crf_sw.anom, ceres_crf_lw.anom, ERA_temp.ts, 3)
ERAT_ERA.monthly<-getRegressionRow(era_crf_net.anom, era_crf_sw.anom, era_crf_lw.anom, ERA_temp.ts, 1)
ERAT_ERA.seasonal<-getRegressionRow(era_crf_net.anom, era_crf_sw.anom, era_crf_lw.anom, ERA_temp.ts, 3)
ERAT_EBAF.monthly<-getRegressionRow(ebaf_crf_net.anom, ebaf_crf_sw.anom, ebaf_crf_lw.anom, ERA_temp.ts, 1)
ERAT_EBAF.seasonal<-getRegressionRow(ebaf_crf_net.anom, ebaf_crf_sw.anom, ebaf_crf_lw.anom, ERA_temp.ts, 3)
ERAT_AIRS.monthly<-getRegressionRowLw(ceres_airs_lw.anom, ERA_temp.ts, 1)
ERAT_AIRS.seasonal<-getRegressionRowLw(ceres_airs_lw.anom, ERA_temp.ts, 3)

#Against ERA-skt
ERASKT_CERES.monthly<-getRegressionRow(ceres_crf_net.anom, ceres_crf_sw.anom, ceres_crf_lw.anom, ERA_skin.ts, 1)
ERASKT_CERES.seasonal<-getRegressionRow(ceres_crf_net.anom, ceres_crf_sw.anom, ceres_crf_lw.anom, ERA_skin.ts, 3)
ERASKT_ERA.monthly<-getRegressionRow(era_crf_net.anom, era_crf_sw.anom, era_crf_lw.anom, ERA_skin.ts, 1)
ERASKT_ERA.seasonal<-getRegressionRow(era_crf_net.anom, era_crf_sw.anom, era_crf_lw.anom, ERA_skin.ts, 3)
ERASKT_EBAF.monthly<-getRegressionRow(ebaf_crf_net.anom, ebaf_crf_sw.anom, ebaf_crf_lw.anom, ERA_skin.ts, 1)
ERASKT_EBAF.seasonal<-getRegressionRow(ebaf_crf_net.anom, ebaf_crf_sw.anom, ebaf_crf_lw.anom, ERA_skin.ts, 3)
ERASKT_AIRS.monthly<-getRegressionRowLw(ceres_airs_lw.anom, ERA_skin.ts, 1)
ERASKT_AIRS.seasonal<-getRegressionRowLw(ceres_airs_lw.anom, ERA_skin.ts, 3)


tblMatrix1<-rbind(c(GISS_CERES.monthly, GISS_CERES.seasonal), c(ERAT_CERES.monthly,ERAT_CERES.seasonal),
		c(ERASKT_CERES.monthly,ERASKT_CERES.seasonal), 
		c(GISS_ERA.monthly, GISS_ERA.seasonal), c(ERAT_ERA.monthly, ERAT_ERA.seasonal),
		c(ERASKT_ERA.monthly, ERASKT_ERA.seasonal), 
		c(GISS_EBAF.monthly, GISS_EBAF.seasonal), c(ERAT_EBAF.monthly, ERAT_EBAF.seasonal),
		c(ERASKT_EBAF.monthly, ERASKT_EBAF.seasonal),
		c(GISS_AIRS.monthly, GISS_AIRS.seasonal), c(ERAT_AIRS.monthly, ERAT_AIRS.seasonal),
		c(ERASKT_AIRS.monthly, ERASKT_AIRS.seasonal))

#Perform kernel adjustments to go from CRF values to R_cld values
ceres_rcloud_net.anom<-anom(doKernelAdjust(ceres_crf_net.anom, TRUE, TRUE, TRUE)) 
ceres_rcloud_sw.anom<-anom(doKernelAdjust(ceres_crf_sw.anom, TRUE, FALSE, TRUE)) 
ceres_rcloud_lw.anom<-anom(doKernelAdjust(ceres_crf_lw.anom, TRUE, TRUE, FALSE)) 
ebaf_rcloud_net.anom<-anom(doKernelAdjust(ebaf_crf_net.anom, TRUE, TRUE, TRUE)) 
ebaf_rcloud_sw.anom<-anom(doKernelAdjust(ebaf_crf_sw.anom, TRUE, FALSE, TRUE)) 
ebaf_rcloud_lw.anom<-anom(doKernelAdjust(ebaf_crf_lw.anom, TRUE, TRUE, FALSE)) 
era_rcloud_net.anom<-anom(doKernelAdjust(era_crf_net.anom, TRUE, TRUE, TRUE)) 
era_rcloud_sw.anom<-anom(doKernelAdjust(era_crf_sw.anom, TRUE, FALSE, TRUE)) 
era_rcloud_lw.anom<-anom(doKernelAdjust(era_crf_lw.anom, TRUE, TRUE, FALSE)) 
airs_rcloud_lw.anom<-anom(doKernelAdjust(ceres_airs_lw.anom, TRUE, TRUE, FALSE))

#Now do regressions against R_cld
GISS_CERES.monthly<-getRegressionRow(ceres_rcloud_net.anom, ceres_rcloud_sw.anom, ceres_rcloud_lw.anom, GISS.ts, 1)
GISS_CERES.seasonal<-getRegressionRow(ceres_rcloud_net.anom, ceres_rcloud_sw.anom, ceres_rcloud_lw.anom, GISS.ts, 3)
GISS_ERA.monthly<-getRegressionRow(era_rcloud_net.anom, era_rcloud_sw.anom, era_rcloud_lw.anom, GISS.ts, 1)
GISS_ERA.seasonal<-getRegressionRow(era_rcloud_net.anom, era_rcloud_sw.anom, era_rcloud_lw.anom, GISS.ts, 3)

ERAT_CERES.monthly<-getRegressionRow(ceres_rcloud_net.anom, ceres_rcloud_sw.anom, ceres_rcloud_lw.anom, ERA_temp.ts, 1)
ERAT_CERES.seasonal<-getRegressionRow(ceres_rcloud_net.anom, ceres_rcloud_sw.anom, ceres_rcloud_lw.anom, ERA_temp.ts, 3)
ERAT_ERA.monthly<-getRegressionRow(era_rcloud_net.anom, era_rcloud_sw.anom, era_rcloud_lw.anom, ERA_temp.ts, 1)
ERAT_ERA.seasonal<-getRegressionRow(era_rcloud_net.anom, era_rcloud_sw.anom, era_rcloud_lw.anom, ERA_temp.ts, 3)

GISS_EBAF.monthly<-getRegressionRow(ebaf_rcloud_net.anom, ebaf_rcloud_sw.anom, ebaf_rcloud_lw.anom, GISS.ts, 1)
GISS_EBAF.seasonal<-getRegressionRow(ebaf_rcloud_net.anom, ebaf_rcloud_sw.anom, ebaf_rcloud_lw.anom, GISS.ts, 3)
ERAT_EBAF.monthly<-getRegressionRow(ebaf_rcloud_net.anom, ebaf_rcloud_sw.anom, ebaf_rcloud_lw.anom, ERA_temp.ts, 1)
ERAT_EBAF.seasonal<-getRegressionRow(ebaf_rcloud_net.anom, ebaf_rcloud_sw.anom, ebaf_rcloud_lw.anom, ERA_temp.ts, 3)

GISS_AIRS.monthly<-getRegressionRowLw(airs_rcloud_lw.anom, GISS.ts, 1)
GISS_AIRS.seasonal<-getRegressionRowLw(airs_rcloud_lw.anom, GISS.ts, 3)
ERAT_AIRS.monthly<-getRegressionRowLw(airs_rcloud_lw.anom, ERA_temp.ts, 1)
ERAT_AIRS.seasonal<-getRegressionRowLw(airs_rcloud_lw.anom, ERA_temp.ts, 3)

ERASKT_CERES.monthly<-getRegressionRow(ceres_rcloud_net.anom, ceres_rcloud_sw.anom, ceres_rcloud_lw.anom, ERA_skin.ts, 1)
ERASKT_CERES.seasonal<-getRegressionRow(ceres_rcloud_net.anom, ceres_rcloud_sw.anom, ceres_rcloud_lw.anom, ERA_skin.ts, 3)
ERASKT_ERA.monthly<-getRegressionRow(era_rcloud_net.anom, era_rcloud_sw.anom, era_rcloud_lw.anom, ERA_skin.ts, 1)
ERASKT_ERA.seasonal<-getRegressionRow(era_rcloud_net.anom, era_rcloud_sw.anom, era_rcloud_lw.anom, ERA_skin.ts, 3)
ERASKT_EBAF.monthly<-getRegressionRow(ebaf_rcloud_net.anom, ebaf_rcloud_sw.anom, ebaf_rcloud_lw.anom, ERA_skin.ts, 1)
ERASKT_EBAF.seasonal<-getRegressionRow(ebaf_rcloud_net.anom, ebaf_rcloud_sw.anom, ebaf_rcloud_lw.anom, ERA_skin.ts, 3)
ERASKT_AIRS.monthly<-getRegressionRowLw(airs_rcloud_lw.anom, ERA_skin.ts, 1)
ERASKT_AIRS.seasonal<-getRegressionRowLw(airs_rcloud_lw.anom, ERA_skin.ts, 3)

tblMatrix2<-rbind(c(GISS_CERES.monthly, GISS_CERES.seasonal), c(ERAT_CERES.monthly,ERAT_CERES.seasonal),
		c(ERASKT_CERES.monthly,ERASKT_CERES.seasonal), 
		c(GISS_ERA.monthly, GISS_ERA.seasonal), c(ERAT_ERA.monthly, ERAT_ERA.seasonal),
		c(ERASKT_ERA.monthly, ERASKT_ERA.seasonal), 
		c(GISS_EBAF.monthly, GISS_EBAF.seasonal), c(ERAT_EBAF.monthly, ERAT_EBAF.seasonal),
		c(ERASKT_EBAF.monthly, ERASKT_EBAF.seasonal),
		c(GISS_AIRS.monthly, GISS_AIRS.seasonal), c(ERAT_AIRS.monthly, ERAT_AIRS.seasonal),
		c(ERASKT_AIRS.monthly, ERASKT_AIRS.seasonal))

#here we merge the two regression tables (R_cld vs. Temp and CRF vs. Temp) with (CRF vs. Temp in parenthesis)
tblMatrix<-array(0, dim=c(nrow(tblMatrix1), ncol(tblMatrix1)))
for (row in 1:nrow(tblMatrix1))
{
	for (col in 1:ncol(tblMatrix1))
	{
		tblMatrix[row,col]<-paste(tblMatrix2[row,col], "  (", tblMatrix1[row,col], ")", sep='')
	}
}

#output results
df<-data.frame(tblMatrix)
rownames(df)[1]<-"GISTemp and CERES_CRF"
rownames(df)[2]<-"ERAT and CERES_CRF"
rownames(df)[3]<-"ERASKT and CERES_CRF"
rownames(df)[4]<-"GISTemp and ERA CRF"
rownames(df)[5]<-"ERAT and ERA_CRF"
rownames(df)[6]<-"ERASKT and ERA_CRF"
rownames(df)[7]<-"GISTemp and EBAF_CRF"
rownames(df)[8]<-"ERAT and EBAF_CRF"
rownames(df)[9]<-"ERASKT and EBAF_CRF"
rownames(df)[10]<-"GISTemp and AIRS_CRF"
rownames(df)[11]<-"ERAT and AIRS_CRF"
rownames(df)[12]<-"ERASKT and AIRS_CRF"

write.table(df, file="CloudFeedback_AQUA_2.txt",sep="\t")

################################
# FIGURE 3: CERES CRF and R_cloud (AIRS and ERA-Interim adjusted)
ceres_rcloud_net_eraAdj.anom<-doKernelAdjust(ceres_crf_net.anom, FALSE, TRUE, TRUE)
ts.plot(ceres_crf_net.anom, ceres_rcloud_net_eraAdj.anom, ceres_rcloud_net.anom, col=c("black", "red", "blue"), 
	ylab=expression(Delta~R[cloud]~','~Delta~CRF~' ('~Wm^{-2}~')'))

legend("bottom", legend=c(expression(Delta~CRF), expression(Delta~R['cloud,ECMWF-adj']), 
	expression(Delta~R['cloud,AIRS-adj'])), col=c("black", "red", "blue"), lty=1)

#calculate effect of AIRS vs. ERA-Interim LW adjustment against different temperature series
test1.lm<-lm(ceres_rcloud_net_eraAdj.anom ~ GISS.ts)
test2.lm<-lm(ceres_rcloud_net.anom ~ GISS.ts)
coef(test1.lm)[2]-coef(test2.lm)[2]
test1.lm<-lm(ceres_rcloud_net_eraAdj.anom ~ ERA_temp.ts)
test2.lm<-lm(ceres_rcloud_net.anom ~ ERA_temp.ts)
coef(test1.lm)[2]-coef(test2.lm)[2]
test1.lm<-lm(ceres_rcloud_net_eraAdj.anom ~ ERA_skin.ts)
test2.lm<-lm(ceres_rcloud_net.anom ~ ERA_skin.ts)
coef(test1.lm)[2]-coef(test2.lm)[2]

###################Difference in water vapor trends (AIRS vs. ERA-Interim), effect on TOA
era_fluxes<-read.table("ERA_GFDL_GlobalFluxContributions_2000_1-2011_11.txt")
airs_fluxes<-read.table("AIRS_GFDL_GlobalFluxContributions_2002_9-2011_10.txt")
era_lwWvFlux_clr<-anom(window(ts(era_fluxes[[9]], start=c(2000,1), freq=12), start=start, end=end))
airs_lwWvFlux_clr<-anom(window(ts(airs_fluxes[[7]], start=c(2002,9), freq=12), end=end))

diff.ts<-airs_lwWvFlux_clr- era_lwWvFlux_clr 
diff_trend.lm<-lm(diff.ts ~ time(diff.ts))

#difference in AIRS 
era_lw_clr.anom<-anom(window(era_lw_clr, start=start, end=end))
airs_lw_clr.anom<-anom(window(clear_sky.airs, start=start, end=end))
diff2.ts<-airs_lw_clr.anom - era_lw_clr.anom
test1.lm<-lm(diff2.ts ~ ERA_temp.ts)
test2.lm<-lm(diff2.ts ~ GISS.ts)
test3.lm<-lm(diff2.ts ~ ERA_skin.ts)


