#Troy Masters 2012

setwd("C:/Climate/Masters_2012/extra") 

#must first install ncdf
library(ncdf)
library(fields)

#first must download ERA-Interim data from http://data-portal.ecmwf.int/data/d/interim_moda/levtype=sfc/
#then get GFDL kernels from Dr. Soden's website http://metofis.rsmas.miami.edu/~bsoden/data/kernels.html

era.nc<-open.ncdf("era_skt_albedo.nc") #open our nc file
#pressure level 100 to 1000, 90 degrees N to -90 degrees N, 0 degrees E to 358.5 degrees E
eraAtmo.nc<-open.ncdf("era_pressure_levels.nc")

#kernel pressure level1=10, etc.
#units for layers are per 100 hPa
#kernel pressure level18=1050, 17=1000, 850, etc.
#units for layers are per 100 hPa!
#kernels 0 degrees E to 258.5 degrees E
#-90 to 90 N
kernel_atmt.nc<-open.ncdf("lw_t_gfdl_std.nc")
kernel_lwwv.nc<-open.ncdf("lw_q_gfdl_std.nc")
kernel_swwv.nc<-open.ncdf("sw_q_gfdl_std.nc")
kernel_albedo.nc<-open.ncdf("sw_a_gfdl_std.nc")
kernel_atmt_clr.nc<-open.ncdf("lwclr_t_gfdl_std.nc")
kernel_lwwv_clr.nc<-open.ncdf("lwclr_q_gfdl_std.nc")
kernel_swwv_clr.nc<-open.ncdf("swclr_q_gfdl_std.nc")
kernel_albedo_clr.nc<-open.ncdf("swclr_a_gfdl_std.nc")

numLon<-144
numLat<-90
numMonths<-143

doRegrid<-function(eraData, startLonCells)
{
	regrid_interim<-array(0, dim=c(startLonCells,numLat))
	for (lon in 1:startLonCells)
	{
		regrid_interim[lon,]<-regrid_interim[lon,]+spline(x=eraData[lon,],n=(numLat))$y
	}
	regrid<-array(0, dim=c(numLon,numLat))
	for (lat in 1:numLat)
	{
		regrid[,lat]<-regrid[,lat]+spline(x=regrid_interim[,lat],n=(numLon))$y
	}
	regrid
}


############determine weight for each gridcell################
globalWeight2<-array(0,dim=c(numLon, numLat))
for (lon in 1:numLon)
{
	for (lat in 1:numLat)
	{
		lat1<-(90-(lat-1)*(180/numLat))*pi/180
		lat2<-(90-(lat)*(180/numLat))*pi/180
		globalWeight2[lon,lat]<-(360/numLon)*pi/180*(sin(lat1) - sin(lat2))
		#Surface Area = R^2(lon1-lon2)(sin (lat1) - sin(lat2))
	}
}
globalWeight2<-globalWeight2/(4*pi)


##############Calculate global temperatures####################
getGlobalAverageTs<-function(data)
{
	globalVar<-numeric(numMonths)
	for (month in 1:numMonths)
	{
		globalVar[month]<-sum(data[,,month]*globalWeight2)
	}
	ts(globalVar, start=c(2000,1), freq=12)
}

calcSurfaceGriddedFluxContributions<-function(kernelNC,kernelVar, divisor, eraNC, eraVar)
{
	fluxContributions<-array(0, dim=c(numLon,numLat,numMonths))
	eraValues<-array(0, dim=c(numLon,numLat,numMonths))
	eraAverages<-array(0, dim=c(numLon,numLat,12))
	
	for (month in 1:numMonths)
	{
		era<-get.var.ncdf(eraNC, eraVar, start=c(1,1,month), count=c(240,121,1))
			
		eraValues[,,month]<-doRegrid(era, 240)
		m_ind<-((month-1) %% 12 + 1)
		if (month <  133)
		{
			eraAverages[,,m_ind]<-eraAverages[,,m_ind] + eraValues[,,month]/11
		}
	}
	
	m_ind<-((month-1) %% 12 + 1)

	for (month in 1:numMonths)
	{
		monIndex<-((month-1) %% 12 + 1)
		kernel<-get.var.ncdf(kernelNC, kernelVar, start=c(1,1,1,monIndex), count=c(144, 90,1,1))
		if (divisor == 1) #actually surface temperatures
		{
			kernel<-get.var.ncdf(kernelNC, kernelVar, start=c(1,1,18,monIndex), count=c(144, 90,1,1))	
		} 
		kernel.regrid<-doRegrid(kernel, 144) / divisor
		era.anom<-eraValues[,,month]-eraAverages[,,monIndex]
		fluxContributions[,,month]<-era.anom[,(numLat):1]*kernel.regrid
	}
	fluxContributions
}

albedoFluxes<-calcSurfaceGriddedFluxContributions(kernel_albedo.nc, "sw_a", .01, era.nc, "fal")
albedoFluxes.ts<-getGlobalAverageTs(albedoFluxes)
albedoFluxes_clr<-calcSurfaceGriddedFluxContributions(kernel_albedo_clr.nc, "sw_a", .01, era.nc, "fal")
albedoFluxes_clr.ts<-getGlobalAverageTs(albedoFluxes_clr)

surfaceTempFluxes<-calcSurfaceGriddedFluxContributions(kernel_atmt.nc, "lw_t", 1, era.nc, "skt")
surfaceTempFluxes.ts<-getGlobalAverageTs(surfaceTempFluxes)
surfaceTempFluxes_clr<-calcSurfaceGriddedFluxContributions(kernel_atmt_clr.nc, "lwclr_t", 1, era.nc, "skt")
surfaceTempFluxes_clr.ts<-getGlobalAverageTs(surfaceTempFluxes_clr)

layerBorders<-numeric(13)
layerBorders[1]<-85
layerBorders[2]<-125
layerBorders[3]<-175
layerBorders[4]<-225
layerBorders[5]<-275
layerBorders[6]<-350
layerBorders[7]<-450
layerBorders[8]<-550
layerBorders[9]<-650
layerBorders[10]<-775
layerBorders[11]<-887.5
layerBorders[12]<-962.5
layerBorders[13]<-1050

pressureLayers<-numeric(12)
pressureLayers[1]<-100
pressureLayers[2]<-150
pressureLayers[3]<-200
pressureLayers[4]<-250
pressureLayers[5]<-300
pressureLayers[6]<-400
pressureLayers[7]<-500
pressureLayers[8]<-600
pressureLayers[9]<-700
pressureLayers[10]<-850
pressureLayers[11]<-925
pressureLayers[12]<-1000


layerThickness<-numeric(12)
for (layer in 1:12)
{
	layerThickness[layer]<-layerBorders[layer+1]-layerBorders[layer]
}

calcAtmoTempGriddedFluxContributions<-function(kernelNC,kernelVar, eraNC, eraVar)
{
	fluxContributions<-array(0, dim=c(numLon,numLat,numMonths))
	eraValues<-array(0, dim=c(numLon,numLat,12, numMonths))
	eraAverages<-array(0, dim=c(numLon,numLat,12, 12))
	
	for (month in 1:numMonths)
	{
		for (level in 1:12)
		{
			era<-get.var.ncdf(eraNC, eraVar, start=c(1,1,level+5, month), count=c(240,121,1,1))
			eraValues[,,level,month]<-doRegrid(era, 240)
		}
		m_ind<-((month-1) %% 12 + 1)
		if (month < 133)
		{
			eraAverages[,,,m_ind]<-eraAverages[,,,m_ind] + eraValues[,,,month]/11
		}
	}

	for (month in 1:numMonths)
	{
		monIndex<-((month-1) %% 12 + 1)
		kernel.regrid<-array(0,dim=c(numLon,numLat,12))

		#do some inverting here to match ERA
		kernel<-get.var.ncdf(kernelNC, kernelVar, start=c(1,1,6,monIndex), count=c(144,90,12,1))
		#kernel.regrid[,,level]<-doRegrid(kernel, 144)
		era.anom<-eraValues[,,,month]-eraAverages[,,,monIndex]
		#fluxAllLevels<-era.anom[,(numLat):1,]*kernel.regrid*layerThickness/100
		fluxAllLevels<-era.anom[,(numLat):1,]*kernel*layerThickness/100
		
		for (y in 1:(numLat))
		{
			lat<-(y-.5)*(180/numLat) - 90
			tropoPause<-100+abs(lat)/90*200
			for (layer in 1:12)
			{
				weight<-0
				if (layerBorders[layer] > tropoPause)
				{	#full layer should be included
					weight<-1.0
				}
				else if (layerBorders[layer+1] < tropoPause)
				{	#none of this layer should be included
					weight<-0
				}
				else
				{
					#part of this layer should be included
					weight<-(layerBorders[layer+1]-tropoPause)/(layerThickness[layer])
				}
				fluxContributions[,y,month]<-fluxContributions[,y,month]+(fluxAllLevels[,y,layer]*weight)
			}
		}
	}
	fluxContributions
}

atmoTempFluxes<-calcAtmoTempGriddedFluxContributions(kernel_atmt.nc,"lw_t", eraAtmo.nc, "t")
atmoTempFluxes.ts<-getGlobalAverageTs(atmoTempFluxes)
atmoTempFluxes_clr<-calcAtmoTempGriddedFluxContributions(kernel_atmt_clr.nc,"lwclr_t", eraAtmo.nc, "t")
atmoTempFluxes_clr.ts<-getGlobalAverageTs(atmoTempFluxes_clr)

satPressure<-function(T)
{
	#exp(77.3450+.0057*T-7235/T)/t^8.2 #K to Pa
	#6.22*exp(17.62*T/(243.12 + T)) #C to hPa
	10^ (  -7.90298*(373.16/T-1) #K to hPa                                                             [1] 
             +5.02808* log10(373.16/T) 
             -1.3816* 10^(-7) *(10^(11.344*(1-T/373.16))-1) 
             + 8.1328*10^(-3)* (10^(-3.49149*(373.16/T-1))-1) 
             + log10(1013.246)  ) 

}

#T in K, P in hPA, RH between 0 and 1
#returns kg/kg
specificHumidity<-function(RH, T, P)
{
	e_s<-satPressure(T) #sat pressure in hPa
	e<-e_s*RH
	eps<-0.622
	q<-(eps*e/(P-(1-eps)*e))
}

q.standardAnomaly<-array(0, dim=c(numLon, numLat, 12, 12))
for (month in 1:(12*4))
{
	m_ind<-((month-1) %% 12 + 1)
	for (level in 1:12)
	{
		era_rh<-get.var.ncdf(eraAtmo.nc, "r", start=c(1,1,level, month), count=c(240,121,1,1))
		era.rh.regrid<-doRegrid(era_rh, 240)/100
		era_T<-get.var.ncdf(eraAtmo.nc, "t", start=c(1,1,level, month), count=c(240,121,1,1))
		era.T.regrid<-doRegrid(era_T, 240)
		#q.temp<-get.var.ncdf(eraAtmo.nc, "q", start=c(1,1,level, month), count=c(240,121,1,1))
		#q.actual[,,level]<-doRegrid(q.temp,240)
		for (x in 1:(numLon))
		{
			for (y in 1:(numLat))
			{
				logPerturbed<-log(specificHumidity(era.rh.regrid[x,y], era.T.regrid[x,y]+1.0,pressureLayers[level]))
				logNormal<-log(specificHumidity(era.rh.regrid[x,y], era.T.regrid[x,y],pressureLayers[level]))
				logDiff<-logPerturbed-logNormal
				#q.calc[x,y,level]<-specificHumidity(era.rh.regrid[x,y], era.T.regrid[x,y],pressureLayers[level])
				if (is.na(logDiff)) logDiff<-9999 #high number so flux contribution will be nothing
				q.standardAnomaly[x,y,level,m_ind]<-q.standardAnomaly[x,y,level,m_ind] + (logDiff)/4
			}
		}
	}
}


calcWaterVaporGriddedFluxContributions<-function(kernelNC,kernelVar, eraNC, eraVar)
{
	fluxContributions<-array(0, dim=c(numLon,numLat,numMonths))
	eraValues<-array(0, dim=c(numLon,numLat,12, numMonths))
	eraAverages<-array(0, dim=c(numLon,numLat,12, 12))
	
	for (month in 1:numMonths)
	{
		for (level in 1:12)
		{
			era<-get.var.ncdf(eraNC, eraVar, start=c(1,1,level+5, month), count=c(240,121,1,1))
			eraValues[,,level,month]<-doRegrid(era, 240)
		}
		m_ind<-((month-1) %% 12 + 1)
		if (month < 133)
		{
			eraAverages[,,,m_ind]<-eraAverages[,,,m_ind] + eraValues[,,,month]/11
		}
	}

	for (month in 1:numMonths)
	{
		monIndex<-((month-1) %% 12 + 1)
		kernel<-get.var.ncdf(kernelNC, kernelVar, start=c(1,1,6,monIndex), count=c(144,90,12,1))
		era.anom<-log(eraValues[,,,month])-log(eraAverages[,,,monIndex]) #log difference
		fluxAllLevels<-era.anom[,(numLat):1,]*kernel/q.standardAnomaly[,(numLat):1,,monIndex]*layerThickness/100		

		for (y in 1:(numLat))
		{
			lat<-(y-.5)*(180/numLat) - 90
			tropoPause<-100+abs(lat)/90*200
			for (layer in 1:12)
			{
				weight<-0
				if (layerBorders[layer] > tropoPause)
				{	#full layer should be included
					weight<-1.0
				}
				else if (layerBorders[layer+1] < tropoPause)
				{	#none of this layer should be included
					weight<-0
				}
				else
				{
					#part of this layer should be included
					weight<-(layerBorders[layer+1]-tropoPause)/(layerThickness[layer])
				}
				fluxContributions[,y,month]<-fluxContributions[,y,month]+(fluxAllLevels[,y,layer]*weight)
			}
		}
	}
	fluxContributions
}


lwWvFluxes<-calcWaterVaporGriddedFluxContributions(kernel_lwwv.nc,"lw_q", eraAtmo.nc, "q")
lwWvFluxes.ts<-getGlobalAverageTs(lwWvFluxes)
lwWvFluxes_clr<-calcWaterVaporGriddedFluxContributions(kernel_lwwv_clr.nc,"lwclr_q", eraAtmo.nc, "q")
lwWvFluxes_clr.ts<-getGlobalAverageTs(lwWvFluxes_clr)

swWvFluxes<-calcWaterVaporGriddedFluxContributions(kernel_swwv.nc,"sw_q", eraAtmo.nc, "q")
swWvFluxes.ts<-getGlobalAverageTs(swWvFluxes)
swWvFluxes_clr<-calcWaterVaporGriddedFluxContributions(kernel_swwv_clr.nc,"swclr_q", eraAtmo.nc, "q")
swWvFluxes_clr.ts<-getGlobalAverageTs(swWvFluxes_clr)

data<-ts.union(albedoFluxes.ts, surfaceTempFluxes.ts, atmoTempFluxes.ts, lwWvFluxes.ts, swWvFluxes.ts, 
	albedoFluxes_clr.ts, surfaceTempFluxes_clr.ts, atmoTempFluxes_clr.ts, lwWvFluxes_clr.ts, swWvFluxes_clr.ts)
write.table(data, "ERA_GFDL_GlobalFluxContributions_2000_1-2011_11.txt")
